/*
 * Copyright (c) 2015 Elvira Khabirova <lineprinter0@gmail.com>
 * Copyright (c) 2015-2016 Dmitry V. Levin <ldv@altlinux.org>
 * Copyright (c) 2015-2019 The strace developers.
 * All rights reserved.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "tests.h"
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/shm.h>

#ifndef SHM_HUGE_SHIFT
# define SHM_HUGE_SHIFT 26
#endif

#ifndef SHM_HUGE_MASK
# define SHM_HUGE_MASK 0x3f
#endif

#ifndef SHM_STAT_ANY
# define SHM_STAT_ANY 15
#endif

#ifndef SHM_NORESERVE
# define SHM_NORESERVE 010000
#endif

#undef TEST_SHMCTL_BOGUS_ADDR
#undef TEST_SHMCTL_BOGUS_CMD

/*
 * Starting with commit glibc-2.32~80, on every 32-bit architecture
 * where 32-bit time_t support is enabled, glibc tries to retrieve
 * the data provided in the third argument of shmctl call.
 */
#if GLIBC_PREREQ_GE(2, 32) && defined __TIMESIZE && __TIMESIZE != 64
# define TEST_SHMCTL_BOGUS_ADDR 0
#endif

/*
 * Starting with commit glibc-2.32.9000-207-g9ebaabeaac1a96b0d91f,
 * glibc skips shmctl syscall invocations and returns EINVAL
 * for invalid shmctl commands.
 * It has been backported into glic-2.28-153 in RHEL 8.5.
 */
#if GLIBC_PREREQ_GE(2, 28)
# define TEST_SHMCTL_BOGUS_CMD 0
#endif

#ifndef TEST_SHMCTL_BOGUS_ADDR
# define TEST_SHMCTL_BOGUS_ADDR 1
#endif
#ifndef TEST_SHMCTL_BOGUS_CMD
# define TEST_SHMCTL_BOGUS_CMD 1
#endif

#include "xlat.h"
#include "xlat/shm_resource_flags.h"

#if XLAT_RAW
# define str_ipc_flags "0x2ce1e00"
# define str_shm_huge "21<<26"
# define str_ipc_private "0"
# define str_ipc_rmid "0"
# define str_ipc_set "0x1"
# define str_ipc_stat "0x2"
# define str_shm_stat "0xd"
# define str_shm_info "0xe"
# define str_shm_stat_any "0xf"
# define str_ipc_64 "0x100"
# define str_bogus_cmd "0xdefaced2"
#elif XLAT_VERBOSE
# define str_ipc_flags \
	"0x2ce1e00 /\\* IPC_CREAT\\|IPC_EXCL\\|SHM_HUGETLB\\|SHM_NORESERVE" \
	"\\|0x2ce0000 \\*/"
# define str_shm_huge "21<<26 /\\* SHM_HUGE_SHIFT \\*/"
# define str_ipc_private "0 /\\* IPC_PRIVATE \\*/"
# define str_ipc_rmid "0 /\\* IPC_RMID \\*/"
# define str_ipc_set "0x1 /\\* IPC_SET \\*/"
# define str_ipc_stat "0x2 /\\* IPC_STAT \\*/"
# define str_shm_stat "0xd /\\* SHM_STAT \\*/"
# define str_shm_info "0xe /\\* SHM_INFO \\*/"
# define str_shm_stat_any "0xf /\\* SHM_STAT_ANY \\*/"
# define str_ipc_64 "0x100 /\\* IPC_64 \\*/"
# define str_bogus_cmd "0xdefaced2 /\\* SHM_\\?\\?\\? \\*/"
#else
# define str_ipc_flags \
	"IPC_CREAT\\|IPC_EXCL\\|SHM_HUGETLB\\|SHM_NORESERVE\\|0x2ce0000"
# define str_shm_huge "21<<SHM_HUGE_SHIFT"
# define str_ipc_private "IPC_PRIVATE"
# define str_ipc_rmid "IPC_RMID"
# define str_ipc_set "IPC_SET"
# define str_ipc_stat "IPC_STAT"
# define str_shm_stat "SHM_STAT"
# define str_shm_info "SHM_INFO"
# define str_shm_stat_any "SHM_STAT_ANY"
# define str_ipc_64 "IPC_64"
# define str_bogus_cmd "0xdefaced2 /\\* SHM_\\?\\?\\? \\*/"
#endif

static int id = -1;

static void
cleanup(void)
{
	shmctl(id, IPC_RMID, NULL);
	printf("shmctl\\(%d, (%s\\|)?%s, NULL\\) = 0\n",
	       id, str_ipc_64, str_ipc_rmid);
	id = -1;
}

int
main(void)
{
	static const key_t private_key =
		(key_t) (0xffffffff00000000ULL | IPC_PRIVATE);
	static const key_t bogus_key = (key_t) 0xeca86420fdb97531ULL;
#if TEST_SHMCTL_BOGUS_CMD || TEST_SHMCTL_BOGUS_ADDR
	static const int bogus_id = 0xdefaced1;
#endif
#if TEST_SHMCTL_BOGUS_CMD
	static const int bogus_cmd = 0xdefaced2;
#endif
#if TEST_SHMCTL_BOGUS_ADDR
	static void * const bogus_addr = (void *) -1L;
#endif
	static const size_t bogus_size =
	/*
	 * musl sets size to SIZE_MAX if size argument is greater than
	 * PTRDIFF_MAX - musl/src/ipc/shmget.c
	 */
#ifdef __GLIBC__
		(size_t) 0xdec0ded1dec0ded2ULL;
#else
		(size_t) 0x1e55c0de5dec0dedULL;
#endif
	static const unsigned int bogus_ipc_shm_flags =
		IPC_CREAT | IPC_EXCL | SHM_HUGETLB | SHM_NORESERVE;
	static const unsigned int huge_mask = SHM_HUGE_MASK << SHM_HUGE_SHIFT;
	static const unsigned int huge_flags = 21 << SHM_HUGE_SHIFT;
	int bogus_flags;
	int rc;
	struct shmid_ds ds;

	rc = shmget(bogus_key, bogus_size, 0);
	printf("shmget\\(%#llx, %zu, 000\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       sprintrc_grep(rc));

	rc = shmget(bogus_key, bogus_size, huge_flags);
	printf("shmget\\(%#llx, %zu, %s\\|%#03o\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       str_shm_huge, 0, sprintrc_grep(rc));

	bogus_flags = 0xface1e55 & ~(bogus_ipc_shm_flags | huge_mask);
	rc = shmget(bogus_key, bogus_size, bogus_flags);
	printf("shmget\\(%#llx, %zu, %#x\\|%#03o\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       bogus_flags & ~0777,
	       bogus_flags & 0777, sprintrc_grep(rc));

	bogus_flags |= bogus_ipc_shm_flags;
	rc = shmget(bogus_key, bogus_size, bogus_flags);
	printf("shmget\\(%#llx, %zu, %s\\|%#03o\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       str_ipc_flags,
	       bogus_flags & 0777, sprintrc_grep(rc));

	bogus_flags |= huge_flags;
	rc = shmget(bogus_key, bogus_size, bogus_flags);
	printf("shmget\\(%#llx, %zu, %s\\|%s\\|%#03o\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       str_ipc_flags, str_shm_huge,
	       bogus_flags & 0777, sprintrc_grep(rc));

	bogus_flags &= ~bogus_ipc_shm_flags;
	rc = shmget(bogus_key, bogus_size, bogus_flags);
	printf("shmget\\(%#llx, %zu, %#x\\|%s\\|%#03o\\) = %s\n",
	       zero_extend_signed_to_ull(bogus_key), bogus_size,
	       bogus_flags & ~(0777 | huge_mask),
	       str_shm_huge,
	       bogus_flags & 0777, sprintrc_grep(rc));

	id = shmget(private_key, 1, 0600);
	if (id < 0)
		perror_msg_and_skip("shmget");
	printf("shmget\\(%s, 1, 0600\\) = %d\n", str_ipc_private, id);
	atexit(cleanup);

#if TEST_SHMCTL_BOGUS_CMD
	rc = shmctl(bogus_id, bogus_cmd, NULL);
	printf("shmctl\\(%d, (%s\\|)?%s, NULL\\) = %s\n",
	       bogus_id, str_ipc_64, str_bogus_cmd, sprintrc_grep(rc));
#endif

#if TEST_SHMCTL_BOGUS_ADDR
	rc = shmctl(bogus_id, IPC_STAT, bogus_addr);
	printf("shmctl\\(%d, (%s\\|)?%s, %p\\) = %s\n",
	       bogus_id, str_ipc_64, str_ipc_stat, bogus_addr,
	       sprintrc_grep(rc));
#endif

	if (shmctl(id, IPC_STAT, &ds))
		perror_msg_and_skip("shmctl IPC_STAT");
	printf("shmctl\\(%d, (%s\\|)?%s, \\{shm_perm=\\{uid=%u, gid=%u, "
		"mode=%#o, key=%u, cuid=%u, cgid=%u\\}, shm_segsz=%u, shm_cpid=%d, "
		"shm_lpid=%d, shm_nattch=%u, shm_atime=%u, shm_dtime=%u, "
		"shm_ctime=%u\\}\\) = 0\n",
		id, str_ipc_64, str_ipc_stat,
		(unsigned) ds.shm_perm.uid, (unsigned) ds.shm_perm.gid,
		(unsigned) ds.shm_perm.mode, (unsigned) ds.shm_perm.__key,
		(unsigned) ds.shm_perm.cuid, (unsigned) ds.shm_perm.cgid,
		(unsigned) ds.shm_segsz, (int) ds.shm_cpid,
		(int) ds.shm_lpid, (unsigned) ds.shm_nattch,
		(unsigned) ds.shm_atime, (unsigned) ds.shm_dtime,
		(unsigned) ds. shm_ctime);

	if (shmctl(id, IPC_SET, &ds))
		perror_msg_and_skip("shmctl IPC_SET");
	printf("shmctl\\(%d, (%s\\|)?%s, \\{shm_perm=\\{uid=%u, gid=%u"
	       ", mode=%#o\\}, ...\\}\\) = 0\n",
	       id, str_ipc_64, str_ipc_set,
	       (unsigned) ds.shm_perm.uid, (unsigned) ds.shm_perm.gid,
	       (unsigned) ds.shm_perm.mode);

	rc = shmctl(0, SHM_INFO, &ds);
	printf("shmctl\\(0, (%s\\|)?%s, %p\\) = %s\n",
	       str_ipc_64, str_shm_info, &ds, sprintrc_grep(rc));

	rc = shmctl(id, SHM_STAT, &ds);
	printf("shmctl\\(%d, (%s\\|)?%s, %p\\) = %s\n",
	       id, str_ipc_64, str_shm_stat, &ds, sprintrc_grep(rc));

	rc = shmctl(id, SHM_STAT_ANY, &ds);
	printf("shmctl\\(%d, (%s\\|)?%s, %p\\) = %s\n",
	       id, str_ipc_64, str_shm_stat_any, &ds, sprintrc_grep(rc));

	return 0;
}
