/*
 * Copyright (c) 2008, Jamie Beverly. 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 * 
 *    1. Redistributions of source code must retain the above copyright notice, this list of
 *       conditions and the following disclaimer.
 * 
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list
 *       of conditions and the following disclaimer in the documentation and/or other materials
 *       provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY Jamie Beverly ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL Jamie Beverly OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are those of the
 * authors and should not be interpreted as representing official policies, either expressed
 * or implied, of Jamie Beverly.
 */


#include "includes.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>

#include "xmalloc.h"
#include "ssh.h"
#include "ssh2.h"
#include "sshbuf.h"
#include "log.h"
#include "compat.h"
#include "sshkey.h"
#include "ssherr.h"
#include "pathnames.h"
#include "misc.h"
#include "secure_filename.h"

#include "identity.h"
#include "pam_user_authorized_keys.h"

#define SSH2_MSG_USERAUTH_TRUST_REQUEST          54

/* extern u_char  *session_id2;
extern uint8_t  session_id_len;
 */

int
userauth_pubkey_from_id(const char *ruser, Identity * id, struct sshbuf * session_id2)
{
    struct sshbuf  *b = NULL;
    char           *pkalg = NULL;
    u_char         *pkblob = NULL, *sig = NULL;
    size_t          blen = 0, slen = 0;
    int             r, authenticated = 0;

    pkalg = (char *) sshkey_ssh_name(id->key);

    /* first test if this key is even allowed */
    if(! pam_user_key_allowed(ruser, id->key))
        goto user_auth_clean_exit_without_buffer;

    if(sshkey_to_blob(id->key, &pkblob, &blen) != 0)
        goto user_auth_clean_exit_without_buffer;

    /* construct packet to sign and test */
    if ((b = sshbuf_new()) == NULL)
        fatal("%s: sshbuf_new failed", __func__);

    if ((r = sshbuf_put_string(b, sshbuf_ptr(session_id2), sshbuf_len(session_id2))) != 0 ||
        (r = sshbuf_put_u8(b, SSH2_MSG_USERAUTH_TRUST_REQUEST)) != 0 ||
        (r = sshbuf_put_cstring(b, ruser)) != 0 ||
        (r = sshbuf_put_cstring(b, "pam_ssh_agent_auth")) != 0 ||
        (r = sshbuf_put_cstring(b, "publickey")) != 0 ||
        (r = sshbuf_put_u8(b, 1)) != 0 ||
        (r = sshbuf_put_cstring(b, pkalg)) != 0 ||
        (r = sshbuf_put_string(b, pkblob, blen)) != 0)
        fatal("%s: buffer error: %s", __func__, ssh_err(r));

    if (ssh_agent_sign(id->ac->fd, id->key, &sig, &slen, sshbuf_ptr(b), sshbuf_len(b), NULL, 0) != 0)
        goto user_auth_clean_exit;

    /* test for correct signature */
    if (sshkey_verify(id->key, sig, slen, sshbuf_ptr(b), sshbuf_len(b), NULL, 0) == 0)
        authenticated = 1;

  user_auth_clean_exit:
    /* if(&b != NULL) */
    sshbuf_free(b);
  user_auth_clean_exit_without_buffer:
    if(sig != NULL)
        free(sig);
    if(pkblob != NULL)
        free(pkblob);
    CRYPTO_cleanup_all_ex_data();
    return authenticated;
}
