/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1997, 2013 Oracle and/or its affiliates.  All rights reserved.
 *
 * $Id$
 */

#include "db_config.h"

#include "db_int.h"
#include <sys/file.h>

#if !defined(HAVE_FCNTL) || !defined(HAVE_FLOCK)
static int __os_filelocking_notsup __P((ENV *));
#endif

/*
 * __check_lock_fn --
 * Parse /proc/locks to see if the file described by 'fn' is locked.
 * Additionally (if 'pid' is not 0) check if the process holding
 * the lock has the same pid value as 'pid'.
 *
 * Returns 0 if a lock on fn is found, 1 if it is not found and -1 on error.
 * PUBLIC: int __check_lock_fn __P((char *, pid_t));
 */

int __check_lock_fn(fn, pid)
    char *fn;
    pid_t pid;
{
    FILE* fp;
    char buffer[PATH_MAX];
    char *token;
    int i, inode;
    struct stat st;
    pid_t lpid = 0;

    if (!fn)
       return -1;

    fp = fopen("/proc/locks", "r");
    if (!fp)
        return -1;

    /* Get the file's inode */
    if (stat(fn, &st)) {
        fclose(fp);
        return -1;
    }

    while (fgets(buffer, sizeof(buffer), fp))
        for (token = strtok(buffer, " "), i = 0; token; token = strtok(NULL, " "), i++) {
            /* Do not parse any other fields */
            if (i > 5)
                break;
            /* Save the PID */
            if (i == 4)
                lpid = atoi(token);
            /* Check the inode */
            else if (i == 5) {
                inode = 0;
                sscanf(token, "%*02x:%*02x:%d", &inode);
                /* Not the inode we are looking for */
                if (inode != st.st_ino)
                    continue;
                /*
                 * We have the correct file.
                 * We are either looking for a specific process or we do not care at all.
                 */
                if (!pid || lpid == pid) {
                    fclose(fp);
                    return 0;
                }
                /* Not the lock we are looking for */
            }
        }
    fclose(fp);
    return 1;
}

/*
 * __rpm_lock_free --
 * Try to look at a lock used by rpm to see if libdb is being
 * updated and it is safe to access its environment files.
 * PUBLIC: int __rpm_lock_free __P((ENV *));
 */

#define RPM_PATH SHAREDSTATEDIR "/rpm"
#define RPMLOCK_PATH RPM_PATH "/.rpm.lock"

int __rpm_lock_free(env)
    ENV *env;
{
    int ret;

    /* No need to check the transaction lock if not in rpm */
    if (strstr(env->db_home, RPM_PATH) == NULL)
        return 1;

    /* Assume it is safe to rebuild if the lock file does not exist */
    if (access(RPMLOCK_PATH, F_OK) && errno == ENOENT)
        return 1;

    ret = __check_lock_fn(RPMLOCK_PATH, 0);
    /* __check_lock_fn can return -1 on failure - return 0 (taken) instead */
    return ret == -1 ? 0: ret;

}

/*
 * __os_fdlock --
 *	Acquire/release a lock on a byte in a file.
 *
 *	The lock modes supported here are:
 *	DB_LOCK_NG	- release the lock
 *	DB_LOCK_READ	- get shared access
 *	DB_LOCK_WRITE	- get exclusive access
 *
 *	Use fcntl()-like semantics most of the time (DB_REGISTER support). Fcntl
 *	supports range locking, but has the additional broken semantics that
 *	closing any of the file's descriptors releases any locks, even if its
 *	other file descriptors remain open. Thanks SYSV & POSIX.
 *	However, if the offset is negative (which is allowed, because POSIX
 *	off_t a signed integer) then use flock() instead.  It has only whole-
 *	file locks, but they persist until explicitly unlocked or the process
 *	exits.
 * PUBLIC: int __os_fdlock __P((ENV *, DB_FH *, off_t, db_lockmode_t, int));
 */
int
__os_fdlock(env, fhp, offset, lockmode, nowait)
	ENV *env;
	DB_FH *fhp;
	off_t offset;
	db_lockmode_t lockmode;
	int nowait;
{
#ifdef HAVE_FCNTL
	DB_ENV *dbenv;
	struct flock fl;
	int ret, t_ret;
	static char *mode_string[DB_LOCK_WRITE + 1] = {
		"unlock",
		"read",
		"write"
	};
	short mode_fcntl[DB_LOCK_WRITE + 1] = {
		F_UNLCK,
		F_RDLCK,
		F_WRLCK
	};
#ifdef HAVE_FLOCK
	short mode_flock[DB_LOCK_WRITE + 1] = {
		LOCK_UN,
		LOCK_SH,
		LOCK_EX
	};
#endif

	dbenv = env == NULL ? NULL : env->dbenv;

	DB_ASSERT(env, F_ISSET(fhp, DB_FH_OPENED) && fhp->fd != -1);
	DB_ASSERT(env, lockmode <= DB_LOCK_WRITE);

	if (dbenv != NULL && FLD_ISSET(dbenv->verbose, DB_VERB_FILEOPS_ALL)) {
		if (offset < 0)
		      __db_msg(env, DB_STR_A("####",
			  "fileops: flock %s %s %s", "%s %s %s"),
			  fhp->name, mode_string[lockmode],
			  nowait ? "nowait" : "");
	      else
		      __db_msg(env, DB_STR_A("0020",
			  "fileops: fcntls %s %s offset %lu", "%s %s %lu"),
			  fhp->name, mode_string[lockmode], (u_long)offset);
	}

	if (offset < 0) {
#ifdef HAVE_FLOCK
	        RETRY_CHK_EINTR_ONLY(flock(fhp->fd,
	            mode_flock[lockmode] | (nowait ? LOCK_NB : 0)), ret);
#else
		ret = __os_filelocking_notsup(env);
#endif
	} else {
	        fl.l_start = offset;
	        fl.l_len = 1;
	        fl.l_whence = SEEK_SET;
	        fl.l_type = mode_fcntl[lockmode];
	           RETRY_CHK_EINTR_ONLY(
	            fcntl(fhp->fd, nowait ? F_SETLK : F_SETLKW, &fl), ret);
	}

	if (offset < 0 && dbenv != NULL &&
	    FLD_ISSET(dbenv->verbose, DB_VERB_FILEOPS_ALL))
		__db_msg(env, DB_STR_A("####",
		    "fileops: flock %s %s %s returns %s", "%s %s %s"),
		    fhp->name, mode_string[lockmode],
		    nowait ? "nowait" : "", db_strerror(ret));

	if (ret == 0)
		return (0);

	if ((t_ret = __os_posix_err(ret)) != EACCES && t_ret != EAGAIN)
		__db_syserr(env, ret, DB_STR("0139", "fcntl"));
	return (t_ret);
#else
	ret = __os_filelocking_notsup(env);
	COMPQUIET(fhp, NULL);
	COMPQUIET(lockmode, 0);
	COMPQUIET(nowait, 0);
	COMPQUIET(offset, 0);
	return (ret)
#endif
}


#if !defined(HAVE_FCNTL) || !defined(HAVE_FLOCK)
/*
 * __os_filelocking_notsup --
 *	Generate an error message if fcntl() or flock() is requested on a
 *	platform that does not support it.
 *
 */
static int
__os_filelocking_notsup(env)
 	ENV *env;
{
	__db_syserr(env, DB_OPNOTSUP, DB_STR("0140",
	    "advisory file locking unavailable"));
	return (DB_OPNOTSUP);
}
#endif
