/* vim: set tabstop=8 shiftwidth=4 softtabstop=4 expandtab smarttab colorcolumn=80: */
/*
 * Copyright (c) 2019 Red Hat, Inc.
 * Author: Sergio Correia <scorreia@redhat.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "util.h"

struct file_list*
new_file_list(void)
{
    struct file_list *fl = malloc(sizeof(*fl));
    if (!fl) {
        return NULL;
    }
    fl->m_files = NULL;
    fl->m_size = 0;

    return fl;
}

void
free_file_list(struct file_list *fl)
{
    if (!fl) {
        return;
    }

    for (size_t i = 0, size = fl->m_size; i < size; i++) {
        free(fl->m_files[i]);
    }
    free(fl->m_files);
    fl->m_size = 0;
    free(fl);
}

void
cleanup_file_list(struct file_list **fl)
{
    if (!fl || !*fl) {
        return;
    }
    free_file_list(*fl);
}

int
file_list_add(struct file_list *fl, const char *filepath)
{
    if (!fl || !filepath) {
        return 0;
    }

    char **new_files = realloc(fl->m_files, sizeof(char *) * (fl->m_size + 1));
    if (!new_files) {
        fprintf(stderr, "Error reallocating memory for the new file\n");
        return 0;
    }

    fl->m_files = new_files;
    fl->m_files[fl->m_size++] = strdup(filepath);
    return 1;
}

int
match_file(const char *file, const char *pattern)
{
    if (!file) {
        return 0;
    }

    if (!pattern) {
        return 1;
    }

    return strstr(file, pattern) != NULL;
}

int
list_files_cmp_func(const void *a, const void *b)
{
    const char *sa = *(const char**)a;
    const char *sb = *(const char**)b;
    return strcmp(sa, sb);
}

struct file_list*
list_files(const char *path, const char *pattern, int ignore_hidden)
{
    struct file_list *fl = new_file_list();
    struct dirent *d;
    DIR *dir = opendir(path);

    if (dir == NULL) {
        return fl;
    }

    while ((d = readdir(dir)) != NULL) {
        if (strcmp(d->d_name, ".") == 0 || strcmp(d->d_name, "..") == 0) {
            continue;
        }

        if (ignore_hidden && d->d_name[0] == '.') {
            continue;
        }

        if (match_file(d->d_name, pattern)) {
            if (!file_list_add(fl, d->d_name)) {
                fprintf(stderr, "Unable to add file %s to file list.\n", d->d_name);
                continue;
            }
        }
    }

    if (fl->m_size > 1) {
        qsort(fl->m_files, fl->m_size, sizeof(char*), list_files_cmp_func);
    }

    closedir(dir);
    return fl;
}


