/* vim: set tabstop=8 shiftwidth=4 softtabstop=4 expandtab smarttab colorcolumn=80: */
/*
 * Copyright (c) 2019 Red Hat, Inc.
 * Author: Sergio Correia <scorreia@redhat.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <jansson.h>
#include <jose/io.h>

struct tang_jwk {
    json_t* m_json;
    char* m_from_file;
    char* m_str;
    char* m_thp;
    char* m_alg;
};

struct tang_jwk_list {
    struct tang_jwk **m_jwk;
    size_t m_size;
};

struct tang_keys_info {
    char *m_jwkdir;
    struct file_list *m_payload_keys;
    struct file_list *m_sign_keys;

    struct tang_jwk_list *m_derive;
    struct tang_jwk_list *m_adv;
    struct tang_jwk *m_default_adv;
};

/* struct tang_jwk. */
struct tang_jwk *new_tang_jwk(const char* /* file */, const char* /* alg */);
struct tang_jwk* new_tang_jwk_from_args(json_t*, const char* /* thp */, const char* /* alg */);
struct tang_jwk* tang_jwk_dup(const struct tang_jwk*);
struct tang_jwk *generate_new_tang_jwk(const char* /* alg */);
void free_tang_jwk(struct tang_jwk*);
void cleanup_tang_jwk(struct tang_jwk **jwk);

/* struct tang_jwk_list. */
struct tang_jwk_list* new_tang_jwk_list(void);
void free_tang_jwk_list(struct tang_jwk_list*);
int tang_jwk_list_add(struct tang_jwk_list*, struct tang_jwk*);
struct tang_jwk* tang_jwk_list_find_thp(const struct tang_jwk_list*, const char*);

char *jwk_thumbprint(const json_t* /* jwk */, const char* /* alg */);
char *jwk_thumbprint_from_file(const char* /* file */, const char* /* alg */);
int valid_for_signing(const char* /* file */);
int valid_for_signing_and_verifying(const char* /* file */);
int valid_for_deriving_keys(const char* /* file */);

struct tang_keys_info* new_tang_keys_info(const char*);
void free_tang_keys_info(struct tang_keys_info*);
void cleanup_tang_keys_info(struct tang_keys_info**);


struct tang_keys_info* read_keys(const char*);

int process_payload(const json_t*, jose_io_t*);
char* process_adv(const json_t*, json_t*, const json_t*);
int prepare_default_adv(struct tang_keys_info*);

size_t hash_alg_size(void);
int check_keys(const char*);
struct tang_jwk* find_adv(const struct tang_keys_info*, const char*);
struct tang_jwk* find_deriving_key(const struct tang_keys_info*, const char*);
int is_hash(const char*);

