/*
 * Copyright 2012 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *      Daniel Kopecek <dkopecek@redhat.com>
 */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/file.h>

#include <linux/fs.h>
#include <linux/fiemap.h>

#ifndef NDEBUG
# define dP(...)				\
    do { int  __tmp_errno = errno;		\
	fprintf(stderr, "DEBUG: "__VA_ARGS__);	\
	errno = __tmp_errno;			\
    } while(0)
#else
# define dP(...) while(0)
#endif

int fextent_apply(int fd, int (*function)(int, struct fiemap_extent *, void *), void *arg)
{
    int ret = -1;
    struct stat st;
    struct fiemap *em;
    uint32_t extent_count, i;

    // lock, sync, stat
    if (flock(fd, LOCK_EX) != 0) {
	dP("flock(%d, LOCK_EX) failed: %s, %d.\n", fd, strerror(errno), errno);
	return -1;
    }
    if (fsync(fd) != 0) {
	dP("fsync(%d) failed: %s, %d.\n", fd, strerror(errno), errno);
	goto exit_1;
    }
    if (fstat(fd, &st) != 0) {
	dP("fstat(%d) failed: %s, %d.\n", fd, strerror(errno), errno);
	goto exit_1;
    }
    
    /*
     * fiemap => get extent count
     */
    em = malloc(sizeof(struct fiemap));

    if (em == NULL) {
	dP("malloc(%zu) returned NULL!\n", sizeof(struct fiemap));
	goto exit_1;
    }

    memset(em, 0, sizeof(struct fiemap));

    em->fm_start = 0;
    em->fm_length = st.st_size;
    em->fm_extent_count = 0;
    em->fm_mapped_extents = 0;
    em->fm_flags = 0;

    if (ioctl(fd, FS_IOC_FIEMAP, em) != 0) {
	dP("FS_IOC_FIEMAP: %s, %d.\n", strerror(errno), errno);
	goto exit_0;
    }

    extent_count = em->fm_mapped_extents;
    free(em);

    /*
     * fiemap => get extents
     */
    em = malloc (sizeof(struct fiemap)
		 + (sizeof(struct fiemap_extent) * extent_count));

    if (em == NULL) {
	dP("malloc(%zu) returned NULL!\n", sizeof(struct fiemap)
	   + (sizeof (struct fiemap_extent) * extent_count));
	goto exit_0;
    }

    memset(em, 0, sizeof(struct fiemap)
	   + (sizeof(struct fiemap_extent) * extent_count));

    em[0].fm_start = 0;
    em[0].fm_length = st.st_size;
    em[0].fm_extent_count = extent_count;
    em[0].fm_flags = 0;

    if (ioctl(fd, FS_IOC_FIEMAP, em) != 0) {
	dP("FS_IOC_FIEMAP: %s, %d.\n", strerror(errno), errno);
	goto exit_0;
    }
   
    for (i = 0; i < extent_count; ++i) {
	// seek to extent start
	if (lseek(fd, em->fm_extents[i].fe_logical, SEEK_SET) == (off_t)-1) {
	    dP("lseek(%d, %llu, SET) failed: %s, %d.\n",
	       fd, em->fm_extents[i].fe_logical, strerror(errno), errno);
	    goto exit_0;
	}

	ret = function(fd, em->fm_extents + i, arg);
	if (ret != 0)
	    goto exit_0;
    }

    ret = 0;
  exit_0:
    // release resources
    free (em);
  exit_1:
    // unlock
    if (flock(fd, LOCK_UN) != 0)
	ret = -1;

    return ret;
}
