/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*- */

/*
 * X Wayland Support
 *
 * Copyright (C) 2013 Intel Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "config.h"

#include "wayland/meta-xwayland.h"
#include "wayland/meta-xwayland-private.h"

#include <errno.h>
#include <glib-unix.h>
#include <glib.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <sys/random.h>
#include <unistd.h>
#include <X11/Xauth.h>

#include "compositor/meta-surface-actor-wayland.h"
#include "meta/main.h"
#include "meta/meta-x11-display.h"
#include "wayland/meta-wayland-actor-surface.h"

enum
{
  XWAYLAND_SURFACE_WINDOW_ASSOCIATED,

  XWAYLAND_SURFACE_LAST_SIGNAL
};

guint xwayland_surface_signals[XWAYLAND_SURFACE_LAST_SIGNAL];

#define META_TYPE_WAYLAND_SURFACE_ROLE_XWAYLAND (meta_wayland_surface_role_xwayland_get_type ())
G_DECLARE_FINAL_TYPE (MetaWaylandSurfaceRoleXWayland,
                      meta_wayland_surface_role_xwayland,
                      META, WAYLAND_SURFACE_ROLE_XWAYLAND,
                      MetaWaylandActorSurface)

struct _MetaWaylandSurfaceRoleXWayland
{
  MetaWaylandActorSurface parent;
};

G_DEFINE_TYPE (MetaWaylandSurfaceRoleXWayland,
               meta_wayland_surface_role_xwayland,
               META_TYPE_WAYLAND_ACTOR_SURFACE)

static int display_number_override = -1;

void
meta_xwayland_associate_window_with_surface (MetaWindow          *window,
                                             MetaWaylandSurface  *surface)
{
  MetaDisplay *display = window->display;

  /* If the window has an existing surface, like if we're
   * undecorating or decorating the window, then we need
   * to detach the window from its old surface.
   */
  if (window->surface)
    {
      meta_wayland_surface_set_window (window->surface, NULL);
      window->surface = NULL;
    }

  if (!meta_wayland_surface_assign_role (surface,
                                         META_TYPE_WAYLAND_SURFACE_ROLE_XWAYLAND,
                                         NULL))
    {
      wl_resource_post_error (surface->resource,
                              WL_DISPLAY_ERROR_INVALID_OBJECT,
                              "wl_surface@%d already has a different role",
                              wl_resource_get_id (surface->resource));
      return;
    }

  window->surface = surface;
  meta_wayland_surface_set_window (surface, window);
  g_signal_emit (surface->role,
                 xwayland_surface_signals[XWAYLAND_SURFACE_WINDOW_ASSOCIATED],
                 0);

  meta_compositor_window_surface_changed (display->compositor, window);

  /* Now that we have a surface check if it should have focus. */
  meta_display_sync_wayland_input_focus (display);
}

static gboolean
associate_window_with_surface_id (MetaXWaylandManager *manager,
                                  MetaWindow          *window,
                                  guint32              surface_id)
{
  struct wl_resource *resource;

  resource = wl_client_get_object (manager->client, surface_id);
  if (resource)
    {
      MetaWaylandSurface *surface = wl_resource_get_user_data (resource);
      meta_xwayland_associate_window_with_surface (window, surface);
      return TRUE;
    }
  else
    return FALSE;
}

void
meta_xwayland_handle_wl_surface_id (MetaWindow *window,
                                    guint32     surface_id)
{
  MetaWaylandCompositor *compositor = meta_wayland_compositor_get_default ();
  MetaXWaylandManager *manager = &compositor->xwayland_manager;

  if (!associate_window_with_surface_id (manager, window, surface_id))
    {
      /* No surface ID yet, schedule this association for whenever the
       * surface is made known.
       */
      meta_wayland_compositor_schedule_surface_association (compositor,
                                                            surface_id, window);
    }
}

gboolean
meta_xwayland_is_xwayland_surface (MetaWaylandSurface *surface)
{
  MetaWaylandCompositor *compositor = meta_wayland_compositor_get_default ();
  MetaXWaylandManager *manager = &compositor->xwayland_manager;

  return wl_resource_get_client (surface->resource) == manager->client;
}

static gboolean
try_display (int      display,
             char   **filename_out,
             int     *fd_out,
             GError **error)
{
  gboolean ret = FALSE;
  char *filename;
  int fd;

  filename = g_strdup_printf ("/tmp/.X%d-lock", display);

 again:
  fd = open (filename, O_WRONLY | O_CLOEXEC | O_CREAT | O_EXCL, 0444);

  if (fd < 0 && errno == EEXIST)
    {
      char pid[11];
      char *end;
      pid_t other;
      int read_bytes;

      fd = open (filename, O_CLOEXEC, O_RDONLY);
      if (fd < 0)
        {
          g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                       "Failed to open lock file %s: %s",
                       filename, g_strerror (errno));
          goto out;
        }

      read_bytes = read (fd, pid, 11);
      if (read_bytes != 11)
        {
          if (read_bytes < 0)
            {
              g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                           "Failed to read lock file %s: %s",
                           filename, g_strerror (errno));
            }
          else
            {
              g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                           "Failed to read lock file %s",
                           filename);
            }
          goto out;
        }
      close (fd);
      fd = -1;

      pid[10] = '\0';
      other = strtol (pid, &end, 0);
      if (end != pid + 10)
        {
          g_set_error (error, G_IO_ERROR, G_IO_ERROR_INVALID_DATA,
                       "Can't parse lock file %s", filename);
          goto out;
        }

      if (kill (other, 0) < 0 && errno == ESRCH)
        {
          /* Process is dead. Try unlinking the lock file and trying again. */
          if (unlink (filename) < 0)
            {
              g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                           "Failed to unlink stale lock file %s: %m", filename);
              goto out;
            }

          goto again;
        }

      g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                   "Lock file %s already occupied", filename);
      goto out;
    }
  else if (fd < 0)
    {
      g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                   "Failed to create lock file %s: %s",
                   filename, g_strerror (errno));
      goto out;
    }

  ret = TRUE;

 out:
  if (!ret)
    {
      g_free (filename);
      filename = NULL;

      if (fd >= 0)
        {
          close (fd);
          fd = -1;
        }
    }

  *filename_out = filename;
  *fd_out = fd;
  return ret;
}

static char *
create_lock_file (int      display,
                  int     *display_out,
                  GError **error)
{
  g_autoptr (GError) local_error = NULL;
  char *filename;
  int fd;
  char pid[12];
  int size;
  int number_of_tries = 0;

  while (!try_display (display, &filename, &fd, &local_error))
    {
      meta_verbose ("Failed to open display %d: %s\n",
                    display, local_error->message);
      g_clear_error (&local_error);

      display++;
      number_of_tries++;

      /* If we can't get a display after 50 times, then something's wrong. Just
       * abort in this case. */
      if (number_of_tries >= 50)
        {
          g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                       "Tried to bind 50 display numbers, giving up");
          return NULL;
        }
    }

  /* Subtle detail: we use the pid of the wayland compositor, not the xserver
   * in the lock file. Another subtlety: snprintf returns the number of bytes
   * it _would've_ written without either the NUL or the size clamping, hence
   * the disparity in size. */
  size = snprintf (pid, 12, "%10d\n", getpid ());
  errno = 0;
  if (size != 11 || write (fd, pid, 11) != 11)
    {
      if (errno != 0)
        {
          g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                       "Failed to write pid to lock file: %s",
                       g_strerror (errno));
        }
      else
        {
          g_set_error (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                       "Failed to write pid to lock file");
        }
      unlink (filename);
      close (fd);
      g_free (filename);
      return NULL;
    }

  close (fd);

  *display_out = display;
  return filename;
}

static int
bind_to_abstract_socket (int        display,
                         GError   **error)
{
  struct sockaddr_un addr;
  socklen_t size, name_size;
  int fd;

  fd = socket (PF_LOCAL, SOCK_STREAM | SOCK_CLOEXEC, 0);
  if (fd < 0)
    {
      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to create socket: %s", g_strerror (errno));
      return -1;
    }

  addr.sun_family = AF_LOCAL;
  name_size = snprintf (addr.sun_path, sizeof addr.sun_path,
                        "%c/tmp/.X11-unix/X%d", 0, display);
  size = offsetof (struct sockaddr_un, sun_path) + name_size;
  if (bind (fd, (struct sockaddr *) &addr, size) < 0)
    {
      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to bind to @%s: %s",
                   addr.sun_path + 1, g_strerror (errno));
      close (fd);
      return -1;
    }

  if (listen (fd, 1) < 0)
    {
      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to listen on abstract socket @%s: %s",
                   addr.sun_path + 1, g_strerror (errno));
      close (fd);
      return -1;
    }

  return fd;
}

static int
bind_to_unix_socket (int      display,
                     GError **error)
{
  struct sockaddr_un addr;
  socklen_t size, name_size;
  int fd;

  fd = socket (PF_LOCAL, SOCK_STREAM | SOCK_CLOEXEC, 0);
  if (fd < 0)
    return -1;

  addr.sun_family = AF_LOCAL;
  name_size = snprintf (addr.sun_path, sizeof addr.sun_path,
                        "/tmp/.X11-unix/X%d", display) + 1;
  size = offsetof (struct sockaddr_un, sun_path) + name_size;
  unlink (addr.sun_path);
  if (bind (fd, (struct sockaddr *) &addr, size) < 0)
    {
      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to bind to %s: %s",
                   addr.sun_path, g_strerror (errno));
      close (fd);
      return -1;
    }

  if (listen (fd, 1) < 0)
    {
      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to listen on %s: %s",
                   addr.sun_path, g_strerror (errno));
      unlink (addr.sun_path);
      close (fd);
      return -1;
    }

  return fd;
}

static void
xserver_died (GObject      *source,
              GAsyncResult *result,
              gpointer      user_data)
{
  GSubprocess *proc = G_SUBPROCESS (source);
  g_autoptr (GError) error = NULL;

  if (!g_subprocess_wait_finish (proc, result, &error))
    {
      if (g_error_matches (error, G_IO_ERROR, G_IO_ERROR_CANCELLED))
        return;

      g_warning ("Failed to finish waiting for Xwayland: %s", error->message);
    }
  else if (!g_subprocess_get_successful (proc))
    g_warning ("X Wayland crashed; exiting");
  else
    {
      /* For now we simply abort if we see the server exit.
       *
       * In the future X will only be loaded lazily for legacy X support
       * but for now it's a hard requirement. */
      g_warning ("Spurious exit of X Wayland server");
    }

  meta_exit (META_EXIT_ERROR);
}

static int
x_io_error (Display *display)
{
  g_warning ("Connection to xwayland lost");
  meta_exit (META_EXIT_ERROR);

  return 0;
}

void
meta_xwayland_override_display_number (int number)
{
  display_number_override = number;
}

static gboolean
ensure_x11_unix_dir (GError **error)
{
  if (mkdir ("/tmp/.X11-unix", 01777) != 0)
    {
      if (errno == EEXIST)
        return TRUE;

      g_set_error (error, G_IO_ERROR, g_io_error_from_errno (errno),
                   "Failed to create directory \"/tmp/.X11-unix\": %s",
                   g_strerror (errno));
      return FALSE;
    }

  return TRUE;
}

static gboolean
choose_xdisplay (MetaXWaylandManager *manager)
{
  g_autoptr (GError) error = NULL;
  int display = 0;
  char *lock_file = NULL;

  if (display_number_override != -1)
    display = display_number_override;
  else if (g_getenv ("RUNNING_UNDER_GDM"))
    display = 1024;

  if (!ensure_x11_unix_dir (&error))
    {
      g_warning ("Failed to ensure X11 socket directory: %s",
                 error->message);
      return FALSE;
    }

  do
    {
      lock_file = create_lock_file (display, &display, &error);
      if (!lock_file)
        {
          g_warning ("Failed to create an X lock file: %s", error->message);
          return FALSE;
        }

      manager->abstract_fd = bind_to_abstract_socket (display, &error);
      if (manager->abstract_fd < 0)
        {
          unlink (lock_file);

          if (g_error_matches (error, G_IO_ERROR, G_IO_ERROR_ADDRESS_IN_USE))
            {
              meta_verbose ("Failed to bind abstract socket: %s\n", error->message);
              g_clear_error (&error);
              display++;
              continue;
            }
          else
            {
              g_warning ("Failed to bind abstract socket: %s", error->message);
              return FALSE;
            }
        }

      manager->unix_fd = bind_to_unix_socket (display, &error);
      if (manager->unix_fd < 0)
        {
          meta_verbose ("Failed to bind unix socket: %s\n", error->message);
          g_clear_error (&error);
          unlink (lock_file);
          close (manager->abstract_fd);
          display++;
          continue;
        }

      break;
    }
  while (1);

  manager->display_index = display;
  manager->display_name = g_strdup_printf (":%d", manager->display_index);
  manager->lock_file = lock_file;

  g_message ("Using X11 display %s for Xwayland", manager->display_name);

  return TRUE;
}

G_DEFINE_AUTOPTR_CLEANUP_FUNC (FILE, fclose)

static gboolean
prepare_auth_file (MetaXWaylandManager *manager)
{
  Xauth auth_entry = { 0 };
  g_autoptr (FILE) fp = NULL;
  char hostname[HOST_NAME_MAX + 1];
  char auth_data[16];
  int fd;

  manager->auth_file = g_build_filename (g_get_user_runtime_dir (),
                                         ".mutter-Xwaylandauth.XXXXXX",
                                         NULL);

  if (gethostname (hostname, HOST_NAME_MAX) < 0)
    g_strlcpy (hostname, "localhost", HOST_NAME_MAX);

  if (getrandom (auth_data, sizeof (auth_data), 0) != sizeof (auth_data))
    {
      g_warning ("Failed to get random data: %s", g_strerror (errno));
      return FALSE;
    }

  auth_entry.family = FamilyLocal;
  auth_entry.address = hostname;
  auth_entry.address_length = strlen (auth_entry.address);
  auth_entry.name = (char *) "MIT-MAGIC-COOKIE-1";
  auth_entry.name_length = strlen (auth_entry.name);
  auth_entry.data = auth_data;
  auth_entry.data_length = sizeof (auth_data);

  fd = g_mkstemp (manager->auth_file);
  if (fd < 0)
    {
      g_warning ("Failed to open Xauthority file: %s", g_strerror (errno));
      return FALSE;
    }

  fp = fdopen (fd, "w+");
  if (!fp)
    {
      g_warning ("Failed to open Xauthority stream: %s", g_strerror (errno));
      close (fd);
      return FALSE;
    }

  if (!XauWriteAuth (fp, &auth_entry))
    {
      g_warning ("Error writing to Xauthority file: %s", g_strerror (errno));
      return FALSE;
    }

  auth_entry.family = FamilyWild;
  if (!XauWriteAuth (fp, &auth_entry))
    {
      g_warning ("Error writing to Xauthority file: %s", g_strerror (errno));
      return FALSE;
    }

  if (fflush (fp) == EOF)
    {
      g_warning ("Error writing to Xauthority file: %s", g_strerror (errno));
      return FALSE;
    }

  return TRUE;
}

static void
add_local_user_to_xhost (Display *xdisplay)
{
  XHostAddress host_entry;
  XServerInterpretedAddress siaddr;

  siaddr.type = (char *) "localuser";
  siaddr.typelength = strlen (siaddr.type);
  siaddr.value = (char *) g_get_user_name();
  siaddr.valuelength = strlen (siaddr.value);

  host_entry.family = FamilyServerInterpreted;
  host_entry.address = (char *) &siaddr;

  XAddHost (xdisplay, &host_entry);
}

static void
xserver_finished_init (MetaXWaylandManager *manager)
{
  /* At this point xwayland is all setup to start accepting
   * connections so we can quit the transient initialization mainloop
   * and unblock meta_wayland_init() to continue initializing mutter.
   * */
  g_main_loop_quit (manager->init_loop);
  g_clear_pointer (&manager->init_loop, g_main_loop_unref);
}

static gboolean
on_displayfd_ready (int          fd,
                    GIOCondition condition,
                    gpointer     user_data)
{
  MetaXWaylandManager *manager = user_data;

  /* The server writes its display name to the displayfd
   * socket when it's ready. We don't care about the data
   * in the socket, just that it wrote something, since
   * that means it's ready. */
  xserver_finished_init (manager);

  return G_SOURCE_REMOVE;
}

gboolean
meta_xwayland_start (MetaXWaylandManager *manager,
                     struct wl_display   *wl_display)
{
  int xwayland_client_fd[2];
  int displayfd[2];
  gboolean started = FALSE;
  g_autoptr(GSubprocessLauncher) launcher = NULL;
  GSubprocessFlags flags;
  GError *error = NULL;

  if (!choose_xdisplay (manager))
    goto out;

  if (!prepare_auth_file (manager))
    goto out;

  /* We want xwayland to be a wayland client so we make a socketpair to setup a
   * wayland protocol connection. */
  if (socketpair (AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC, 0, xwayland_client_fd) < 0)
    {
      g_warning ("xwayland_client_fd socketpair failed\n");
      goto out;
    }

  if (socketpair (AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC, 0, displayfd) < 0)
    {
      g_warning ("displayfd socketpair failed\n");
      goto out;
    }

  /* xwayland, please. */
  flags = G_SUBPROCESS_FLAGS_NONE;

  if (getenv ("XWAYLAND_STFU"))
    {
      flags |= G_SUBPROCESS_FLAGS_STDOUT_SILENCE;
      flags |= G_SUBPROCESS_FLAGS_STDERR_SILENCE;
    }

  launcher = g_subprocess_launcher_new (flags);

  g_subprocess_launcher_take_fd (launcher, xwayland_client_fd[1], 3);
  g_subprocess_launcher_take_fd (launcher, manager->abstract_fd, 4);
  g_subprocess_launcher_take_fd (launcher, manager->unix_fd, 5);
  g_subprocess_launcher_take_fd (launcher, displayfd[1], 6);

  g_subprocess_launcher_setenv (launcher, "WAYLAND_SOCKET", "3", TRUE);

  /* Use the -terminate parameter to ensure that Xwayland exits cleanly
   * after the last client disconnects. Fortunately that includes the window
   * manager so it won't exit prematurely either. This ensures that Xwayland
   * won't try to reconnect and crash, leaving uninteresting core dumps. We do
   * want core dumps from Xwayland but only if a real bug occurs...
   */
  manager->proc = g_subprocess_launcher_spawn (launcher, &error,
                                               XWAYLAND_PATH, manager->display_name,
                                               "-rootless",
                                               "-terminate",
                                               "-accessx",
                                               "-core",
                                               "-auth", manager->auth_file,
                                               "-listen", "4",
                                               "-listen", "5",
                                               "-displayfd", "6",
                                               NULL);
  if (!manager->proc)
    {
      g_error ("Failed to spawn Xwayland: %s", error->message);
      goto out;
    }

  manager->xserver_died_cancellable = g_cancellable_new ();
  g_subprocess_wait_async (manager->proc, manager->xserver_died_cancellable,
                           xserver_died, NULL);
  g_unix_fd_add (displayfd[0], G_IO_IN, on_displayfd_ready, manager);
  manager->client = wl_client_create (wl_display, xwayland_client_fd[0]);

  /* We need to run a mainloop until we know xwayland has a binding
   * for our xserver interface at which point we can assume it's
   * ready to start accepting connections. */
  manager->init_loop = g_main_loop_new (NULL, FALSE);
  g_main_loop_run (manager->init_loop);

  started = TRUE;

out:
  if (!started)
    {
      unlink (manager->lock_file);
      g_clear_pointer (&manager->lock_file, g_free);
    }
  return started;
}

static void
on_x11_display_closing (MetaDisplay *display)
{
  Display *xdisplay = meta_x11_display_get_xdisplay (display->x11_display);

  meta_xwayland_shutdown_selection (xdisplay);
}

/* To be called right after connecting */
void
meta_xwayland_complete_init (MetaDisplay *display,
                             Display     *xdisplay)
{
  /* We install an X IO error handler in addition to the child watch,
     because after Xlib connects our child watch may not be called soon
     enough, and therefore we won't crash when X exits (and most important
     we won't reset the tty).
  */
  XSetIOErrorHandler (x_io_error);

  g_signal_connect (display, "x11-display-closing",
                    G_CALLBACK (on_x11_display_closing), NULL);
  meta_xwayland_init_selection (xdisplay);
  add_local_user_to_xhost (xdisplay);
}

void
meta_xwayland_stop (MetaXWaylandManager *manager)
{
  char path[256];

  g_cancellable_cancel (manager->xserver_died_cancellable);
  g_clear_object (&manager->proc);
  g_clear_object (&manager->xserver_died_cancellable);

  snprintf (path, sizeof path, "/tmp/.X11-unix/X%d", manager->display_index);
  unlink (path);

  g_clear_pointer (&manager->display_name, g_free);
  if (manager->auth_file)
    {
      unlink (manager->auth_file);
      g_clear_pointer (&manager->auth_file, g_free);
    }
  if (manager->lock_file)
    {
      unlink (manager->lock_file);
      g_clear_pointer (&manager->lock_file, g_free);
    }
}

static MetaWaylandSurface *
xwayland_surface_get_toplevel (MetaWaylandSurfaceRole *surface_role)
{
  return meta_wayland_surface_role_get_surface (surface_role);
}

static void
xwayland_surface_sync_actor_state (MetaWaylandActorSurface *actor_surface)
{
  MetaWaylandSurfaceRole *surface_role =
    META_WAYLAND_SURFACE_ROLE (actor_surface);
  MetaWaylandSurface *surface =
    meta_wayland_surface_role_get_surface (surface_role);
  MetaWaylandActorSurfaceClass *actor_surface_class =
    META_WAYLAND_ACTOR_SURFACE_CLASS (meta_wayland_surface_role_xwayland_parent_class);

  if (surface->window)
    actor_surface_class->sync_actor_state (actor_surface);
}

static void
meta_wayland_surface_role_xwayland_init (MetaWaylandSurfaceRoleXWayland *role)
{
}

static void
meta_wayland_surface_role_xwayland_class_init (MetaWaylandSurfaceRoleXWaylandClass *klass)
{
  MetaWaylandSurfaceRoleClass *surface_role_class =
    META_WAYLAND_SURFACE_ROLE_CLASS (klass);
  MetaWaylandActorSurfaceClass *actor_surface_class =
    META_WAYLAND_ACTOR_SURFACE_CLASS (klass);

  surface_role_class->get_toplevel = xwayland_surface_get_toplevel;

  actor_surface_class->sync_actor_state = xwayland_surface_sync_actor_state;

  xwayland_surface_signals[XWAYLAND_SURFACE_WINDOW_ASSOCIATED] =
    g_signal_new ("window-associated",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0, NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}
