/* SPDX-License-Identifier: MIT */
/*
 * Description: test io_uring ioctl
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "liburing.h"

#ifndef BLKDISCARD
#define BLKDISCARD     _IO(0x12,119)
#endif

typedef unsigned long long ull;
char *dev_path;

static int init_params(int cmd, int *fd, int *len, void **buf)
{
	*len = cmd;
	switch (cmd) {
	case BLKDISCARD:
		*fd = open(dev_path, O_RDWR);
		ull *range = (ull *)malloc(sizeof(ull) * 2);
		range[0] = 0;
		range[1] = 1024;
		*buf = range;
		break;
	default:
		return -1;
	}

	return 0;
}

int ioctl_clean(int cmd, int fd, void *buf)
{
	int ret;
	if (fd > 0) {
		ret = close(fd);
		if (ret)
			return ret;
	}

	switch (cmd) {
	case BLKDISCARD:
		if (buf)
			free(buf);
		break;
	}

	return 0;
}
static int ioctl_extra_check(int cmd, void *buf)
{
	switch (cmd) {
	case BLKDISCARD:
		break;
	}
	return 0;
}

static int test_ioctl(struct io_uring *ring, int cmd)
{
	struct io_uring_cqe *cqe;
	struct io_uring_sqe *sqe;
	int fd = 0, len, ret, ret2;
	void *buf = NULL;

	ret = init_params(cmd, &fd, &len, &buf);
	if (ret == -1) {
		perror("invalid command");
		goto err;
	}
	if (fd < 0) {
		perror("open");
		goto err;
	}

	sqe = io_uring_get_sqe(ring);
	if (!sqe) {
		fprintf(stderr, "get sqe failed\n");
		goto err;
	}
	io_uring_prep_ioctl(sqe, fd, len, buf);

	ret = io_uring_submit(ring);
	if (ret <= 0) {
		fprintf(stderr, "sqe submit failed: %d\n", ret);
		goto err;
	}

	ret = io_uring_wait_cqe(ring, &cqe);
	if (ret < 0) {
		fprintf(stderr, "wait completion %d\n", ret);
		goto err;
	}

	ret = 1;
	switch (cqe->res) {
	/*
	 * Two cases return -EINVAL:
	 *      - sys_ioctl() itself returns it
	 *      - IORING_OP_IOCTL isn't supported in io_uring
	 *
	 * To distinguish them, one solution is to return other stuff other
	 * than -EINVAL when sys_ioctl() logic returns -EINVAL. But this
	 * change the logic of the original sys_ioctl() syscall.
	 * Leave it as it is for now.
	 */
	case -EINVAL:
		fprintf(stderr, "cqe->res is -EINVAL\n");
		break;
	case -EFAULT:
		fprintf(stderr, "cqe->res is -EFAULT\n");
		break;
	case -EBADF:
		fprintf(stderr, "cqe->res is -EBADF\n");
		break;
	case -ENOTTY:
		fprintf(stderr, "cqe->res is -ENOTTY\n");
		break;
	default:
		if (cqe->res) {
			fprintf(stderr, "cqe->res is %d\n", cqe->res);
		} else if (ioctl_extra_check(cmd, buf)) {
			fprintf(stderr, "ioctl cmd:%d failed\n", cmd);
		} else {
			ret = 0;
		}
	}

	io_uring_cqe_seen(ring, cqe);
err:
	ret2 = ioctl_clean(cmd, fd, buf);
	/*
	 * it's not a test error, just print warning, still return 0
	 */
	if (ret2)
		fprintf(stderr, "close test file failure\n");

	return ret;
}

int main(int argc, char *argv[])
{
	struct io_uring ring;
	int ret;

	if (argc < 2) {
		fprintf(stderr, "no ssd device indicated, skip.\n");
		return 0;
	}

	dev_path = argv[1];

	ret = io_uring_queue_init(8, &ring, 0);
	if (ret) {
		fprintf(stderr, "ring setup failed\n");
		return 1;
	}

	ret = test_ioctl(&ring, BLKDISCARD);
	if (ret) {
		fprintf(stderr, "test_ioctl BLKDISCARD failed\n");
		return ret;
	}

	return 0;
}

