
/* Desktop Icons GNOME Shell extension
 *
 * Copyright (C) 2017 Carlos Soriano <csoriano@redhat.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

const { Gio, GObject, Gdk, Gtk } = imports.gi;
const ExtensionUtils = imports.misc.extensionUtils;
const Gettext = imports.gettext.domain('gnome-shell-extensions');
const _ = Gettext.gettext;
const N_ = e => e;
const cssData = `
   .no-border {
       border: none;
   }

   .border {
       border: 1px solid;
       border-radius: 3px;
       border-color: #b6b6b3;
       box-shadow: inset 0 0 0 1px rgba(74, 144, 217, 0);
       background-color: white;
   }

   .margins {
       padding-left: 8px;
       padding-right: 8px;
       padding-bottom: 8px;
   }

   .message-label {
       font-weight: bold;
   }
`;

var settings;

function init() {
    settings = ExtensionUtils.getSettings("org.gnome.shell.extensions.heads-up-display");
    let cssProvider = new Gtk.CssProvider();
    cssProvider.load_from_data(cssData);

    let screen = Gdk.Screen.get_default();
    Gtk.StyleContext.add_provider_for_screen(screen, cssProvider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
}

function buildPrefsWidget() {
    ExtensionUtils.initTranslations();

    let contents = new Gtk.Box({
        orientation: Gtk.Orientation.VERTICAL,
        border_width: 20,
        spacing: 10,
    });

    contents.add(buildSwitch('show-when-locked', _("Show message when screen is locked")));
    contents.add(buildSwitch('show-when-unlocking', _("Show message on unlock screen")));
    contents.add(buildSwitch('show-when-unlocked', _("Show message when screen is unlocked")));
    contents.add(buildSpinButton('idle-timeout', _("Seconds after user goes idle before reshowing message")));

    let outerMessageBox = new Gtk.Box({
        orientation: Gtk.Orientation.VERTICAL,
        border_width: 0,
        spacing: 5,
    });
    contents.add(outerMessageBox);

    let messageLabel = new Gtk.Label({
        label: 'Message',
        halign: Gtk.Align.START,
    });
    messageLabel.get_style_context().add_class("message-label");
    outerMessageBox.add(messageLabel);

    let innerMessageBox = new Gtk.Box({
        orientation: Gtk.Orientation.VERTICAL,
        border_width: 0,
        spacing: 0,
    });
    innerMessageBox.get_style_context().add_class("border");
    outerMessageBox.add(innerMessageBox);

    innerMessageBox.add(buildEntry('message-heading', _("Message Heading")));
    innerMessageBox.add(buildTextView('message-body', _("Message Body")));
    contents.show_all();
    return contents;
}

function buildTextView(key, labelText) {
    let textView = new Gtk.TextView({
        accepts_tab: false,
        wrap_mode: Gtk.WrapMode.WORD,
    });
    settings.bind(key, textView.get_buffer(), 'text', Gio.SettingsBindFlags.DEFAULT);

    let scrolledWindow = new Gtk.ScrolledWindow({
        expand: true,
    });
    let styleContext = scrolledWindow.get_style_context();
    styleContext.add_class("margins");

    scrolledWindow.add(textView);
    return scrolledWindow;
}
function buildEntry(key, labelText) {
    let entry = new Gtk.Entry({ placeholder_text: labelText });
    let styleContext = entry.get_style_context();
    styleContext.add_class("no-border");
    settings.bind(key, entry, 'text', Gio.SettingsBindFlags.DEFAULT);

    entry.get_settings()['gtk-entry-select-on-focus'] = false;

    return entry;
}

function buildSpinButton(key, labelText) {
    let hbox = new Gtk.Box({
        orientation: Gtk.Orientation.HORIZONTAL,
        spacing: 10,
    });
    let label = new Gtk.Label({
        label: labelText,
        xalign: 0,
    });
    let adjustment = new Gtk.Adjustment({
        value: 0,
        lower: 0,
        upper: 2147483647,
        step_increment: 1,
        page_increment: 60,
        page_size: 60,
    });
    let spinButton = new Gtk.SpinButton({
        adjustment: adjustment,
        climb_rate: 1.0,
        digits: 0,
        max_width_chars: 3,
        width_chars: 3,
    });
    settings.bind(key, spinButton, 'value', Gio.SettingsBindFlags.DEFAULT);
    hbox.pack_start(label, true, true, 0);
    hbox.add(spinButton);
    return hbox;
}

function buildSwitch(key, labelText) {
    let hbox = new Gtk.Box({
        orientation: Gtk.Orientation.HORIZONTAL,
        spacing: 10,
    });
    let label = new Gtk.Label({
        label: labelText,
        xalign: 0,
    });
    let switcher = new Gtk.Switch({
        active: settings.get_boolean(key),
    });
    settings.bind(key, switcher, 'active', Gio.SettingsBindFlags.DEFAULT);
    hbox.pack_start(label, true, true, 0);
    hbox.add(switcher);
    return hbox;
}
