#ifndef HMAC_H
#define HMAC_H

#include "hash-method.h"
#include "sha1.h"
#include <openssl/objects.h>
#include <openssl/hmac.h>
#include <openssl/kdf.h>
#include <openssl/err.h>

#define HMAC_MAX_CONTEXT_SIZE 256

struct openssl_hmac_context_priv {
#ifdef HAVE_HMAC_CTX_NEW
	HMAC_CTX *ctx;
#else
	HMAC_CTX ctx;
#endif
	const struct hash_method *hash;
};

struct orig_hmac_context_priv {
	char ctx[HMAC_MAX_CONTEXT_SIZE];
	char ctxo[HMAC_MAX_CONTEXT_SIZE];
	const struct hash_method *hash;
};

struct openssl_hmac_context {
	union {
		struct openssl_hmac_context_priv priv;
		uint64_t padding_requirement;
	} u;
};

struct orig_hmac_context {
	union {
		struct orig_hmac_context_priv priv;
		uint64_t padding_requirement;
	} u;
};

void openssl_hmac_init(struct openssl_hmac_context *ctx, const unsigned char *key,
		size_t key_len, const struct hash_method *meth);
void openssl_hmac_final(struct openssl_hmac_context *ctx, unsigned char *digest);

static inline void
openssl_hmac_update(struct openssl_hmac_context *_ctx, const void *data, size_t size)
{
	struct openssl_hmac_context_priv *ctx = &_ctx->u.priv;
	HMAC_Update(ctx->ctx, data, size);
/*	if (ec != 1)
    {
        const char *ebuf = NULL;
        const char **error_r = &ebuf;
		dcrypt_openssl_error(error_r);
    }*/
}

void orig_hmac_init(struct orig_hmac_context *ctx, const unsigned char *key,
		size_t key_len, const struct hash_method *meth);
void orig_hmac_final(struct orig_hmac_context *ctx, unsigned char *digest);


static inline void
orig_hmac_update(struct orig_hmac_context *_ctx, const void *data, size_t size)
{
	struct orig_hmac_context_priv *ctx = &_ctx->u.priv;

	ctx->hash->loop(ctx->ctx, data, size);
}

buffer_t *openssl_t_hmac_data(const struct hash_method *meth,
		      const unsigned char *key, size_t key_len,
		      const void *data, size_t data_len);
buffer_t *openssl_t_hmac_buffer(const struct hash_method *meth,
			const unsigned char *key, size_t key_len,
			const buffer_t *data);
buffer_t *openssl_t_hmac_str(const struct hash_method *meth,
		     const unsigned char *key, size_t key_len,
		     const char *data);

void openssl_hmac_hkdf(const struct hash_method *method,
	       const unsigned char *salt, size_t salt_len,
	       const unsigned char *ikm, size_t ikm_len,
	       const unsigned char *info, size_t info_len,
	       buffer_t *okm_r, size_t okm_len);

static inline buffer_t *
openssl_t_hmac_hkdf(const struct hash_method *method,
	    const unsigned char *salt, size_t salt_len,
	    const unsigned char *ikm, size_t ikm_len,
	    const unsigned char *info, size_t info_len,
	    size_t okm_len)
{
	buffer_t *okm_buffer = t_buffer_create(okm_len);
	openssl_hmac_hkdf(method, salt, salt_len, ikm, ikm_len, info, info_len,
		  okm_buffer, okm_len);
	return okm_buffer;
}

#endif
