/*
 * HMAC (RFC-2104) implementation.
 *
 * Copyright (c) 2004 Andrey Panin <pazke@donpac.ru>
 * Copyright (c) 2011-2016 Florian Zeitz <florob@babelmonkeys.de>
 *
 * This software is released under the MIT license.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "lib.h"
#include "hmac.h"
#include "safe-memset.h"
#include "buffer.h"

#include "hex-binary.h"

#ifndef HAVE_HMAC_CTX_NEW
#  define HMAC_Init_ex(ctx, key, key_len, md, impl) \
	HMAC_Init_ex(&(ctx), key, key_len, md, impl)
#  define HMAC_Update(ctx, data, len) HMAC_Update(&(ctx), data, len)
#  define HMAC_Final(ctx, md, len) HMAC_Final(&(ctx), md, len)
#  define HMAC_CTX_free(ctx) HMAC_cleanup(&(ctx))
#else
#  define HMAC_CTX_free(ctx) \
	STMT_START { HMAC_CTX_free(ctx); (ctx) = NULL; } STMT_END
#endif


void openssl_hmac_init(struct openssl_hmac_context *_ctx, const unsigned char *key,
		size_t key_len, const struct hash_method *meth)
{
 	struct openssl_hmac_context_priv *ctx = &_ctx->u.priv;

	const EVP_MD *md;
	const char *ebuf = NULL;
	const char **error_r = &ebuf;

	md = EVP_get_digestbyname(meth->name);
	if(md == NULL) {
		if (error_r != NULL) {
			*error_r = t_strdup_printf("Invalid digest %s",
						   meth->name);
		}
		//return FALSE;
	}

// 	int ec;

	i_assert(md != NULL);
#ifdef HAVE_HMAC_CTX_NEW
	ctx->ctx = HMAC_CTX_new();
/*	if (ctx->ctx == NULL)
		dcrypt_openssl_error(error_r);*/
#endif
	/*ec = */HMAC_Init_ex(ctx->ctx, key, key_len, md, NULL);
}

void orig_hmac_init(struct orig_hmac_context *_ctx, const unsigned char *key,
		size_t key_len, const struct hash_method *meth)
{
    static int no_fips = -1;
    if (no_fips == -1) {
        int fd = open("/proc/sys/crypto/fips_enabled", O_RDONLY);
        if (fd != -1)
        {
            char buf[4];
            if (read(fd, buf, 4) > 0)
            {
                no_fips = buf[0] == '0';   
            }
            close(fd);   
        }
    }
    i_assert(no_fips);
	struct orig_hmac_context_priv *ctx = &_ctx->u.priv;
	int i;
	unsigned char k_ipad[64];
	unsigned char k_opad[64];
	unsigned char hashedkey[meth->digest_size];

	i_assert(meth->context_size <= HMAC_MAX_CONTEXT_SIZE);

	ctx->hash = meth;

	if (key_len > 64) {
		meth->init(ctx->ctx);
		meth->loop(ctx->ctx, key, key_len);
		meth->result(ctx->ctx, hashedkey);
		key = hashedkey;
		key_len = meth->digest_size;
	}

	memcpy(k_ipad, key, key_len);
	memset(k_ipad + key_len, 0, 64 - key_len);
	memcpy(k_opad, k_ipad, 64);

	for (i = 0; i < 64; i++) {
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	meth->init(ctx->ctx);
	meth->loop(ctx->ctx, k_ipad, 64);
	meth->init(ctx->ctxo);
	meth->loop(ctx->ctxo, k_opad, 64);

	safe_memset(k_ipad, 0, 64);
	safe_memset(k_opad, 0, 64);
}

void openssl_hmac_final(struct openssl_hmac_context *_ctx, unsigned char *digest)
{
	int ec;
	unsigned char buf[HMAC_MAX_MD_CBLOCK];
	unsigned int outl;
//     const char *ebuf = NULL;
//     const char **error_r = &ebuf;

    struct openssl_hmac_context_priv *ctx = &_ctx->u.priv;
	ec = HMAC_Final(ctx->ctx, buf, &outl);
	HMAC_CTX_free(ctx->ctx);
	if (ec == 1)
		memcpy(digest, buf, outl);
//	else
//		dcrypt_openssl_error(error_r);

}

void orig_hmac_final(struct orig_hmac_context *_ctx, unsigned char *digest)
{
	struct orig_hmac_context_priv *ctx = &_ctx->u.priv;

	ctx->hash->result(ctx->ctx, digest);

	ctx->hash->loop(ctx->ctxo, digest, ctx->hash->digest_size);
	ctx->hash->result(ctx->ctxo, digest);
}

buffer_t *openssl_t_hmac_data(const struct hash_method *meth,
		      const unsigned char *key, size_t key_len,
		      const void *data, size_t data_len)
{
	struct openssl_hmac_context ctx;
	i_assert(meth != NULL);
	i_assert(key != NULL && key_len > 0);
	i_assert(data != NULL || data_len == 0);

	buffer_t *res = t_buffer_create(meth->digest_size);
	openssl_hmac_init(&ctx, key, key_len, meth);
	if (data_len > 0)
		openssl_hmac_update(&ctx, data, data_len);
	unsigned char *buf = buffer_get_space_unsafe(res, 0, meth->digest_size);
	openssl_hmac_final(&ctx, buf);
	return res;
}

buffer_t *openssl_t_hmac_buffer(const struct hash_method *meth,
			const unsigned char *key, size_t key_len,
			const buffer_t *data)
{
	return openssl_t_hmac_data(meth, key, key_len, data->data, data->used);
}

buffer_t *openssl_t_hmac_str(const struct hash_method *meth,
		     const unsigned char *key, size_t key_len,
		     const char *data)
{
	return openssl_t_hmac_data(meth, key, key_len, data, strlen(data));
}

void openssl_hmac_hkdf(const struct hash_method *method,
	       const unsigned char *salt, size_t salt_len,
	       const unsigned char *ikm, size_t ikm_len,
	       const unsigned char *info, size_t info_len,
	       buffer_t *okm_r, size_t okm_len)
{
	const EVP_MD *md;
	EVP_PKEY_CTX *pctx;
	int r = 1;

	i_assert(method != NULL);
	i_assert(okm_len < 255*method->digest_size);

	/* salt and info can be NULL */
	i_assert(salt != NULL || salt_len == 0);
	i_assert(info != NULL || info_len == 0);

	i_assert(ikm != NULL && ikm_len > 0);
	i_assert(okm_r != NULL && okm_len > 0);


	md = EVP_get_digestbyname(method->name);
	pctx = EVP_PKEY_CTX_new_id(EVP_PKEY_HKDF, NULL);
	unsigned char *okm_buf = buffer_get_space_unsafe(okm_r, 0, okm_len);

	if ((r=EVP_PKEY_derive_init(pctx)) <= 0)
		goto out;
	if ((r=EVP_PKEY_CTX_set_hkdf_md(pctx, md)) <= 0)
		goto out;
	if ((r=EVP_PKEY_CTX_set1_hkdf_salt(pctx, salt, salt_len)) <= 0)
		goto out;
	if ((r=EVP_PKEY_CTX_set1_hkdf_key(pctx, ikm, ikm_len)) <= 0)
		goto out;
	if ((r=EVP_PKEY_CTX_add1_hkdf_info(pctx, info, info_len)) <= 0)
		goto out;
	if ((r=EVP_PKEY_derive(pctx, okm_buf, &okm_len)) <= 0)
		goto out;

     out:
	EVP_PKEY_CTX_free(pctx);
	if (r <= 0) {
		unsigned long ec = ERR_get_error();
		unsigned char *error = t_strdup_printf("%s", ERR_error_string(ec, NULL));
		i_error("%s", error);
	}

}
