/* ====================================================================
 * Copyright (c) 2003-2007 The OpenSSL Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit. (http://www.openssl.org/)"
 *
 * 4. The names "OpenSSL Toolkit" and "OpenSSL Project" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    openssl-core@openssl.org.
 *
 * 5. Products derived from this software may not be called "OpenSSL"
 *    nor may "OpenSSL" appear in their names without prior written
 *    permission of the OpenSSL Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the OpenSSL Project
 *    for use in the OpenSSL Toolkit (http://www.openssl.org/)"
 *
 * THIS SOFTWARE IS PROVIDED BY THE OpenSSL PROJECT ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE OpenSSL PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <string.h>
#include <openssl/err.h>
#ifdef OPENSSL_FIPS
# include <openssl/fips.h>
#endif
#include <openssl/rsa.h>
#include <openssl/evp.h>
#include <openssl/bn.h>
#include <openssl/opensslconf.h>

#ifdef OPENSSL_FIPS

static const unsigned char n[] =
    "\x00\xc9\xd5\x6d\x9d\x90\xdb\x43\xd6\x02\xed\x96\x88\x13\x8a"
    "\xb2\xbf\x6e\xa1\x06\x10\xb2\x78\x37\xa7\x14\xa8\xff\xdd\x00"
    "\xdd\xb4\x93\xa0\x45\xcc\x96\x90\xed\xad\xa9\xdd\xc4\xd6\xca"
    "\x0c\xf0\xed\x4f\x72\x5e\x21\x49\x9a\x18\x12\x15\x8f\x90\x5a"
    "\xdb\xb6\x33\x99\xa3\xe6\xb4\xf0\xc4\x97\x21\x26\xbb\xe3\xba"
    "\xf2\xff\xa0\x72\xda\x89\x63\x8e\x8b\x3e\x08\x9d\x92\x2a\xbe"
    "\x16\xe1\x43\x15\xfc\x57\xc7\x1f\x09\x11\x67\x1c\xa9\x96\xd1"
    "\x8b\x3e\x80\x93\xc1\x59\xd0\x6d\x39\xf2\xac\x95\xcc\x10\x75"
    "\xe9\x31\x24\xd1\x43\xaf\x68\x52\x4b\xe7\x16\xd7\x49\x65\x6f"
    "\x26\xc0\x86\xad\xc0\x07\x0a\xc1\xe1\x2f\x87\x85\x86\x3b\xdc"
    "\x5a\x99\xbe\xe9\xf9\xb9\xe9\x82\x27\x51\x04\x15\xab\x06\x0e"
    "\x76\x5a\x28\x8d\x92\xbd\xc5\xb5\x7b\xa8\xdf\x4e\x47\xa2\xc1"
    "\xe7\x52\xbf\x47\xf7\x62\xe0\x3a\x6f\x4d\x6a\x4d\x4e\xd4\xb9"
    "\x59\x69\xfa\xb2\x14\xc1\xee\xe6\x2f\x95\xcd\x94\x72\xae\xe4"
    "\xdb\x18\x9a\xc4\xcd\x70\xbd\xee\x31\x16\xb7\x49\x65\xac\x40"
    "\x19\x0e\xb5\x6d\x83\xf1\x36\xbb\x08\x2f\x2e\x4e\x92\x62\xa4"
    "\xff\x50\xdb\x20\x45\xa2\xeb\x16\x7a\xf2\xd5\x28\xc1\xfd\x4e" "\x03\x71";

static int corrupt_rsa;

static int setrsakey(RSA *key)
{
    static const unsigned char e[] = "\x01\x00\x01";

    static const unsigned char d[] =
        "\x36\x27\x3d\xb1\xf9\x1b\xdb\xa7\xa0\x41\x7f\x12\x23\xac\x23"
        "\x29\x99\xd5\x3a\x7b\x60\x67\x41\x07\x63\x53\xb4\xd2\xe7\x58"
        "\x95\x0a\xc7\x05\xf3\x4e\xb2\xb4\x12\xd4\x70\xdc\x4f\x85\x06"
        "\xd3\xdd\xd8\x63\x27\x3e\x67\x31\x21\x24\x39\x04\xbc\x06\xa4"
        "\xcc\xce\x2b\x7a\xfe\x7b\xad\xde\x11\x6e\xa3\xa5\xe6\x04\x53"
        "\x0e\xa3\x4e\x2d\xb4\x8f\x31\xbf\xca\x75\x25\x52\x02\x85\xde"
        "\x3d\xb2\x72\x43\xb2\x89\x8a\x9a\x34\x41\x26\x3f\x9a\x67\xbe"
        "\xa4\x96\x7b\x0e\x75\xba\xa6\x93\xd5\xb8\xd8\xb8\x57\xf2\x4b"
        "\x0f\x14\x81\xd1\x57\x4e\xf6\x45\x4c\xa6\x3b\xd0\x70\xca\xd3"
        "\x9d\x55\xde\x22\x05\xe7\x8e\x28\x4d\xee\x11\xcf\xb6\x67\x76"
        "\x09\xd3\xe3\x3c\x13\xf9\x99\x34\x10\x7b\xec\x81\x38\xf0\xb6"
        "\x34\x9c\x9b\x50\x6f\x0b\x91\x81\x4d\x89\x94\x04\x7b\xf0\x3c"
        "\xf4\xb1\xb2\x00\x48\x8d\x5a\x8f\x88\x9e\xc5\xab\x3a\x9e\x44"
        "\x3f\x54\xe7\xd9\x6e\x47\xaa\xa1\xbd\x40\x46\x31\xf9\xf0\x34"
        "\xb6\x04\xe1\x2b\x5b\x73\x86\xdd\x3a\x92\x1b\x71\xc7\x3f\x32"
        "\xe5\xc3\xc2\xab\xa1\x7e\xbf\xa4\x52\xa0\xb0\x68\x90\xd1\x20"
        "\x12\x79\xe9\xd7\xc9\x40\xba\xf2\x19\xc7\xa5\x00\x92\x86\x0d" "\x01";

    static const unsigned char p[] =
        "\x00\xfc\x5c\x6e\x16\xce\x1f\x03\x7b\xcd\xf7\xb3\x72\xb2\x8f"
        "\x16\x72\xb8\x56\xae\xf7\xcd\x67\xd8\x4e\x7d\x07\xaf\xd5\x43"
        "\x26\xc3\x35\xbe\x43\x8f\x4e\x2f\x1c\x43\x4e\x6b\xd2\xb2\xec"
        "\x52\x6d\x97\x52\x2b\xcc\x5c\x3a\x6b\xf4\x14\xc6\x74\xda\x66"
        "\x38\x1c\x7a\x3f\x84\x2f\xe3\xf9\x5a\xb8\x65\x69\x46\x06\xa3"
        "\x37\x79\xb2\xa1\x5b\x58\xed\x5e\xa7\x5f\x8c\x65\x66\xbb\xd1"
        "\x24\x36\xe6\x37\xa7\x3d\x49\x77\x8a\x8c\x34\xd8\x69\x29\xf3"
        "\x4d\x58\x22\xb0\x51\x24\xb6\x40\xa8\x86\x59\x0a\xb7\xba\x5c"
        "\x97\xda\x57\xe8\x36\xda\x7a\x9c\xad";

    static const unsigned char q[] =
        "\x00\xcc\xbe\x7b\x09\x69\x06\xee\x45\xbf\x88\x47\x38\xa8\xf8"
        "\x17\xe5\xb6\xba\x67\x55\xe3\xe8\x05\x8b\xb8\xe2\x53\xd6\x8e"
        "\xef\x2c\xe7\x4f\x4a\xf7\x4e\x26\x8d\x85\x0b\x3f\xec\xc3\x1c"
        "\xd4\xeb\xec\x6a\xc8\x72\x2a\x25\x7d\xfd\xa6\x77\x96\xf0\x1e"
        "\xcd\x28\x57\xf8\x37\x30\x75\x6b\xbd\xd4\x7b\x0c\x87\xc5\x6c"
        "\x87\x40\xa5\xbb\x27\x2c\x78\xc9\x74\x5a\x54\x5b\x0b\x30\x6f"
        "\x44\x4a\xfa\x71\xe4\x21\x61\x66\xf9\xee\x65\xde\x7c\x04\xd7"
        "\xfd\xa9\x15\x5b\x7f\xe2\x7a\xba\x69\x86\x72\xa6\x06\x8d\x9b"
        "\x90\x55\x60\x9e\x4c\x5d\xa9\xb6\x55";

    static const unsigned char dmp1[] =
        "\x7a\xd6\x12\xd0\x0e\xec\x91\xa9\x85\x8b\xf8\x50\xf0\x11\x2e"
        "\x00\x11\x32\x40\x60\x66\x1f\x11\xee\xc2\x75\x27\x65\x4b\x16"
        "\x67\x16\x95\xd2\x14\xc3\x1d\xb3\x48\x1f\xb7\xe4\x0b\x2b\x74"
        "\xc3\xdb\x50\x27\xf9\x85\x3a\xfa\xa9\x08\x23\xc1\x65\x3d\x34"
        "\x3a\xc8\x56\x7a\x65\x45\x36\x6e\xae\x2a\xce\x9f\x43\x43\xd7"
        "\x10\xe9\x9e\x18\xf4\xa4\x35\xda\x8a\x6b\xb0\x3f\xdd\x53\xe3"
        "\xa8\xc5\x4e\x79\x9d\x1f\x51\x8c\xa2\xca\x66\x3c\x6a\x2a\xff"
        "\x8e\xd2\xf3\xb7\xcb\x82\xda\xde\x2c\xe6\xd2\x8c\xb3\xad\xb6"
        "\x4c\x95\x55\x76\xbd\xc9\xc8\xd1";

    static const unsigned char dmq1[] =
        "\x00\x83\x23\x1d\xbb\x11\x42\x17\x2b\x25\x5a\x2c\x03\xe6\x75"
        "\xc1\x18\xa8\xc9\x0b\x96\xbf\xba\xc4\x92\x91\x80\xa5\x22\x2f"
        "\xba\x91\x90\x36\x01\x56\x15\x00\x2c\x74\xa2\x97\xf7\x15\xa1"
        "\x49\xdf\x32\x35\xd2\xdd\x0c\x91\xa6\xf8\xe7\xbe\x81\x36\x9b"
        "\x03\xdc\x6b\x3b\xd8\x5d\x79\x57\xe0\xe6\x4f\x49\xdf\x4c\x5c"
        "\x0e\xe5\x21\x41\x95\xfd\xad\xff\x9a\x3e\xa0\xf9\x0f\x59\x9e"
        "\x6a\xa7\x7b\x71\xa7\x24\x9a\x36\x52\xae\x97\x20\xc1\x5e\x78"
        "\xd9\x47\x8b\x1e\x67\xf2\xaf\x98\xe6\x2d\xef\x10\xd7\xf1\xab"
        "\x49\xee\xe5\x4b\x7e\xae\x1f\x1d\x61";

    static const unsigned char iqmp[] =
        "\x23\x96\xc1\x91\x17\x5e\x0a\x83\xd2\xdc\x7b\x69\xb2\x59\x1d"
        "\x33\x58\x52\x3f\x18\xc7\x09\x50\x1c\xb9\xa1\xbb\x4c\xa2\x38"
        "\x40\x4c\x9a\x8e\xfe\x9c\x90\x92\xd0\x71\x9f\x89\x99\x50\x91"
        "\x1f\x34\x8b\x74\x53\x11\x11\x4a\x70\xe2\xf7\x30\xd8\x8c\x80"
        "\xe1\xcc\x9f\xf1\x63\x17\x1a\x7d\x67\x29\x4c\xcb\x4e\x74\x7b"
        "\xe0\x3e\x9e\x2f\xf4\x67\x8f\xec\xb9\x5c\x00\x1e\x7e\xa2\x7b"
        "\x92\xc9\x6f\x4c\xe4\x0e\xf9\x48\x63\xcd\x50\x22\x5d\xbf\xb6"
        "\x9d\x01\x33\x6a\xf4\x50\xbe\x86\x98\x4f\xca\x3f\x3a\xfa\xcf"
        "\x07\x40\xc4\xaa\xad\xae\xbe\xbf";

    key->n = BN_bin2bn(n, sizeof(n) - 1, key->n);
    if (corrupt_rsa)
        BN_set_bit(key->n, 2048);
    key->e = BN_bin2bn(e, sizeof(e) - 1, key->e);
    key->d = BN_bin2bn(d, sizeof(d) - 1, key->d);
    key->p = BN_bin2bn(p, sizeof(p) - 1, key->p);
    key->q = BN_bin2bn(q, sizeof(q) - 1, key->q);
    key->dmp1 = BN_bin2bn(dmp1, sizeof(dmp1) - 1, key->dmp1);
    key->dmq1 = BN_bin2bn(dmq1, sizeof(dmq1) - 1, key->dmq1);
    key->iqmp = BN_bin2bn(iqmp, sizeof(iqmp) - 1, key->iqmp);
    return 1;
}

void FIPS_corrupt_rsa()
{
    corrupt_rsa = 1;
}

/* Known Answer Test (KAT) data for the above RSA private key signing
 * kat_tbs.
 */

static const unsigned char kat_tbs[] =
    "OpenSSL FIPS 140-2 Public Key RSA KAT";

static const unsigned char kat_RSA_PSS_SHA1[] = {
    0xC2, 0x80, 0x82, 0x56, 0xD8, 0xA7, 0xB2, 0x9C, 0xF5, 0xD6, 0x3C, 0xE3,
    0xBF, 0xE9, 0x3A, 0x53, 0x40, 0xAE, 0xF2, 0xA9, 0x6A, 0x39, 0x49, 0x5B,
    0x05, 0x7F, 0x67, 0x38, 0x2E, 0x1D, 0xE1, 0x93, 0x22, 0x65, 0x79, 0x84,
    0x68, 0xFA, 0xD8, 0xAF, 0xA1, 0x98, 0x61, 0x6F, 0x44, 0x27, 0xA6, 0x8B,
    0xCF, 0x0E, 0x13, 0xA9, 0xCE, 0xD7, 0x6C, 0xD2, 0x38, 0xB5, 0x16, 0xB9,
    0x66, 0x94, 0x48, 0xDE, 0x9E, 0x19, 0x3D, 0x6F, 0xB3, 0xA1, 0x9A, 0x19,
    0xDF, 0xFB, 0xAB, 0xA5, 0x9F, 0x38, 0xDA, 0xC9, 0x21, 0x8F, 0xCE, 0x98,
    0x01, 0x3A, 0xC8, 0xE0, 0xDF, 0xDA, 0xFC, 0xF0, 0xA6, 0x86, 0x29, 0xB5,
    0x7F, 0x61, 0xFB, 0xBA, 0xC5, 0x49, 0xB2, 0x7C, 0x6A, 0x26, 0x82, 0xC4,
    0x8F, 0xAA, 0x5B, 0x10, 0xD5, 0xEE, 0xA0, 0x55, 0x42, 0xEF, 0x32, 0x5A,
    0x3F, 0x55, 0xB3, 0x2C, 0x22, 0xE9, 0x65, 0xDA, 0x8D, 0x0A, 0xB9, 0x70,
    0x43, 0xCC, 0x3F, 0x64, 0x9C, 0xB5, 0x65, 0x49, 0xBD, 0x7F, 0x35, 0xC1,
    0x20, 0x85, 0x24, 0xFE, 0xAA, 0x6B, 0x37, 0x04, 0xA1, 0x0E, 0x9D, 0x5C,
    0xBA, 0x7F, 0x14, 0x69, 0xC5, 0x93, 0xB2, 0x33, 0xC2, 0xC0, 0xC7, 0xDF,
    0x7E, 0x9E, 0xA4, 0xB0, 0xA0, 0x64, 0xD2, 0xAC, 0xFC, 0xFD, 0xFD, 0x99,
    0x8F, 0x6A, 0x40, 0x26, 0xC1, 0x2E, 0x4E, 0x8B, 0x33, 0xBE, 0xF1, 0x45,
    0x59, 0x8F, 0x33, 0x40, 0x1D, 0x2A, 0xD2, 0xF7, 0x50, 0x83, 0x89, 0xCF,
    0x94, 0xC6, 0xF8, 0x36, 0xF0, 0x84, 0x0B, 0x85, 0xA5, 0x02, 0xA9, 0x0F,
    0x41, 0x7A, 0x77, 0xA3, 0x2F, 0x47, 0x1E, 0x1D, 0xEC, 0xE6, 0xD3, 0x01,
    0x1E, 0x6F, 0x7A, 0x96, 0x50, 0x37, 0x37, 0x4B, 0x27, 0x52, 0x0B, 0xDC,
    0xDB, 0xC7, 0xA9, 0x31, 0xB2, 0x40, 0xEE, 0x60, 0x41, 0x26, 0x6A, 0x05,
    0xCE, 0x08, 0x1D, 0x89
};

static const unsigned char kat_RSA_PSS_SHA224[] = {
    0xB4, 0x01, 0x93, 0x16, 0x05, 0xF6, 0xEB, 0xE2, 0xA4, 0xEB, 0x48, 0xAA,
    0x00, 0xF4, 0xA1, 0x99, 0x0A, 0xB4, 0xB6, 0x63, 0xE9, 0x68, 0xCA, 0xB3,
    0x13, 0xD7, 0x66, 0x6A, 0xCD, 0xCB, 0x33, 0x9F, 0xE5, 0x84, 0xE2, 0xC3,
    0x0B, 0x53, 0xE5, 0x8B, 0x96, 0x4B, 0xDB, 0x2D, 0x80, 0xA4, 0x1D, 0xE3,
    0x81, 0xDC, 0x52, 0x99, 0xBA, 0x9B, 0x6A, 0x9D, 0x48, 0x1F, 0x73, 0xF7,
    0xAC, 0x09, 0x13, 0xA1, 0x16, 0x2C, 0x60, 0xFB, 0xBC, 0x25, 0xF7, 0x53,
    0xD1, 0x04, 0x5A, 0x3F, 0x95, 0x09, 0x5E, 0xE5, 0xA2, 0x7D, 0xFC, 0x2A,
    0x51, 0x1D, 0x21, 0xCE, 0x2B, 0x4E, 0x1B, 0xB8, 0xCB, 0xDD, 0x24, 0xEE,
    0x99, 0x1D, 0x37, 0xDC, 0xED, 0x5F, 0x2F, 0x48, 0x5E, 0x33, 0x94, 0x06,
    0x19, 0xCD, 0x5A, 0x26, 0x85, 0x77, 0x9D, 0xAF, 0x86, 0x97, 0xC9, 0x08,
    0xD5, 0x81, 0x0E, 0xB8, 0x9F, 0xB6, 0xAF, 0x20, 0x72, 0xDC, 0x13, 0x4D,
    0x7A, 0xE4, 0x5C, 0x81, 0xDE, 0xC0, 0x3D, 0x19, 0x9C, 0x33, 0x11, 0x07,
    0xD5, 0xA9, 0x51, 0x67, 0xCD, 0xFD, 0x37, 0x61, 0x14, 0x9F, 0xE7, 0x70,
    0x18, 0x32, 0xC3, 0x34, 0x54, 0x0D, 0x4F, 0xB4, 0xAE, 0x9F, 0xEC, 0x64,
    0xD8, 0xB2, 0x16, 0xA4, 0xB2, 0x99, 0x92, 0xCB, 0x7F, 0x1F, 0x06, 0x17,
    0x5F, 0xA1, 0x07, 0x68, 0xAE, 0xA7, 0x2D, 0x03, 0x91, 0x2A, 0x9D, 0x69,
    0xC2, 0x9D, 0x90, 0xF7, 0xF9, 0x66, 0x5D, 0x13, 0xB7, 0x7F, 0xD3, 0x97,
    0x45, 0x97, 0x43, 0xD8, 0xCE, 0x3C, 0xF2, 0x98, 0x98, 0xDD, 0xE2, 0x2D,
    0xCF, 0xA1, 0xC4, 0x25, 0x46, 0x2E, 0xD2, 0xE5, 0x5F, 0xC6, 0x01, 0xC5,
    0x4F, 0x42, 0x2B, 0xDE, 0x0F, 0xEA, 0x4A, 0x4F, 0xC3, 0x5B, 0xDF, 0x9B,
    0x5D, 0x30, 0x18, 0x93, 0xD0, 0xDE, 0xC5, 0x09, 0xAA, 0x57, 0x57, 0xBD,
    0x2D, 0x84, 0x03, 0xB7
};

static const unsigned char kat_RSA_PSS_SHA256[] = {
    0x38, 0xDA, 0x99, 0x51, 0x26, 0x38, 0xC6, 0x7F, 0xC4, 0x81, 0x57, 0x19,
    0x35, 0xC6, 0xF6, 0x1E, 0x90, 0x47, 0x20, 0x55, 0x47, 0x56, 0x26, 0xE9,
    0xF2, 0xA8, 0x39, 0x6C, 0xD5, 0xCD, 0xCB, 0x55, 0xFC, 0x0C, 0xC5, 0xCB,
    0xF7, 0x40, 0x17, 0x3B, 0xCF, 0xE4, 0x05, 0x03, 0x3B, 0xA0, 0xB2, 0xC9,
    0x0D, 0x5E, 0x48, 0x3A, 0xE9, 0xAD, 0x28, 0x71, 0x7D, 0x8F, 0x89, 0x16,
    0x59, 0x93, 0x35, 0xDC, 0x4D, 0x7B, 0xDF, 0x84, 0xE4, 0x68, 0xAA, 0x33,
    0xAA, 0xDC, 0x66, 0x50, 0xC8, 0xA9, 0x32, 0x12, 0xDC, 0xC6, 0x90, 0x49,
    0x0B, 0x75, 0xFF, 0x9B, 0x95, 0x00, 0x9A, 0x90, 0xE0, 0xD4, 0x0E, 0x67,
    0xAB, 0x3C, 0x47, 0x36, 0xC5, 0x2E, 0x1C, 0x46, 0xF0, 0x2D, 0xD3, 0x8B,
    0x42, 0x08, 0xDE, 0x0D, 0xB6, 0x2C, 0x86, 0xB0, 0x35, 0x71, 0x18, 0x6B,
    0x89, 0x67, 0xC0, 0x05, 0xAD, 0xF4, 0x1D, 0x62, 0x4E, 0x75, 0xEC, 0xD6,
    0xC2, 0xDB, 0x07, 0xB0, 0xB6, 0x8D, 0x15, 0xAD, 0xCD, 0xBF, 0xF5, 0x60,
    0x76, 0xAE, 0x48, 0xB8, 0x77, 0x7F, 0xC5, 0x01, 0xD9, 0x29, 0xBB, 0xD6,
    0x17, 0xA2, 0x20, 0x5A, 0xC0, 0x4A, 0x3B, 0x34, 0xC8, 0xB9, 0x39, 0xCF,
    0x06, 0x89, 0x95, 0x6F, 0xC7, 0xCA, 0xC4, 0xE4, 0x43, 0xDF, 0x5A, 0x23,
    0xE2, 0x89, 0xA3, 0x38, 0x78, 0x31, 0x38, 0xC6, 0xA4, 0x6F, 0x5F, 0x73,
    0x5A, 0xE5, 0x9E, 0x09, 0xE7, 0x6F, 0xD4, 0xF8, 0x3E, 0xB7, 0xB0, 0x56,
    0x9A, 0xF3, 0x65, 0xF0, 0xC2, 0xA6, 0x8A, 0x08, 0xBA, 0x44, 0xAC, 0x97,
    0xDE, 0xB4, 0x16, 0x83, 0xDF, 0xE3, 0xEE, 0x71, 0xFA, 0xF9, 0x51, 0x50,
    0x14, 0xDC, 0xFD, 0x6A, 0x82, 0x20, 0x68, 0x64, 0x7D, 0x4E, 0x82, 0x68,
    0xD7, 0x45, 0xFA, 0x6A, 0xE4, 0xE5, 0x29, 0x3A, 0x70, 0xFB, 0xE4, 0x62,
    0x2B, 0x31, 0xB9, 0x7D
};

static const unsigned char kat_RSA_PSS_SHA384[] = {
    0x99, 0x02, 0xC9, 0x1E, 0x31, 0x82, 0xB4, 0xE6, 0x1B, 0x32, 0xCE, 0x5D,
    0x41, 0x1D, 0x00, 0x2F, 0x04, 0x8B, 0xBD, 0x37, 0x79, 0xCF, 0x77, 0x03,
    0x05, 0x6A, 0x21, 0xC7, 0x8D, 0x24, 0x60, 0x49, 0x39, 0x58, 0xC5, 0x27,
    0x8F, 0xC5, 0x97, 0x4A, 0xB2, 0xE1, 0xD4, 0x36, 0x57, 0xBD, 0x43, 0xCC,
    0x7B, 0xCE, 0xF2, 0xA5, 0x30, 0xF8, 0x72, 0x14, 0xBB, 0xD0, 0x9F, 0xC1,
    0x49, 0xC8, 0x1C, 0xAF, 0xCD, 0x95, 0x78, 0x72, 0x25, 0xF9, 0x45, 0xC6,
    0x5B, 0x62, 0x5E, 0x01, 0xD7, 0x40, 0x5E, 0xC8, 0xCA, 0x0A, 0xF3, 0xBA,
    0x08, 0x07, 0x88, 0xCA, 0x49, 0x36, 0x84, 0x7D, 0xF6, 0xFC, 0x5A, 0xDB,
    0xFC, 0x50, 0xD3, 0xEB, 0x3D, 0x83, 0xB0, 0xF5, 0x94, 0x5E, 0x88, 0xC3,
    0x82, 0xCD, 0x53, 0x40, 0x96, 0x18, 0x6B, 0x4A, 0x6C, 0x9C, 0xFE, 0xE5,
    0x3B, 0x75, 0xF9, 0xEB, 0xA5, 0x77, 0x11, 0xEF, 0x88, 0x1C, 0x25, 0x70,
    0x7D, 0x88, 0x5D, 0xC3, 0xCA, 0xE1, 0x49, 0x14, 0x90, 0xAD, 0xF2, 0x5E,
    0x49, 0xD7, 0x99, 0xA5, 0x7B, 0x77, 0x3B, 0x8E, 0xB8, 0xDB, 0xF1, 0x4C,
    0xD6, 0x9A, 0xDC, 0xE5, 0x7A, 0x1C, 0xE1, 0xCE, 0x9D, 0xF1, 0xF3, 0xA0,
    0x0A, 0x35, 0x52, 0x9D, 0xB9, 0x46, 0x94, 0x82, 0x0F, 0xF7, 0xB2, 0x62,
    0x51, 0x70, 0x75, 0xD2, 0x37, 0x96, 0x67, 0x2F, 0xD0, 0x22, 0xD8, 0x07,
    0x8D, 0x69, 0x9E, 0x6D, 0x0B, 0x40, 0x4F, 0x70, 0xEC, 0x0B, 0xCA, 0x88,
    0x80, 0x8D, 0x9A, 0xF4, 0xF9, 0x18, 0x50, 0x27, 0x08, 0xFA, 0xCC, 0xC7,
    0x3F, 0xE4, 0x84, 0x83, 0xA1, 0xB6, 0x1D, 0x23, 0x34, 0xFE, 0x48, 0xE5,
    0xE3, 0xAE, 0x4D, 0x98, 0xBC, 0xA6, 0x8A, 0x9F, 0xFD, 0x4D, 0xDB, 0x9D,
    0xF7, 0xEB, 0x4E, 0xB6, 0x6F, 0x25, 0xEA, 0x7A, 0xE9, 0x85, 0xB2, 0xEF,
    0x90, 0xD2, 0xA6, 0x2B
};

static const unsigned char kat_RSA_PSS_SHA512[] = {
    0x3F, 0x83, 0x43, 0x78, 0x25, 0xBE, 0x81, 0xB2, 0x6E, 0x78, 0x11, 0x32,
    0xD0, 0x88, 0x05, 0x53, 0x95, 0xED, 0x81, 0x12, 0xCE, 0x50, 0xD9, 0x06,
    0x42, 0x89, 0xA0, 0x55, 0x7A, 0x05, 0x13, 0x94, 0x35, 0x9B, 0xCA, 0x5D,
    0xCB, 0xB2, 0x32, 0xE1, 0x04, 0x99, 0xEC, 0xE7, 0xA6, 0x69, 0x4D, 0x2B,
    0xC1, 0x57, 0x13, 0x48, 0x0D, 0x6B, 0x4D, 0x83, 0x28, 0x06, 0x79, 0x9D,
    0xB4, 0x70, 0xCE, 0xC0, 0xFC, 0x3B, 0x69, 0xB3, 0x91, 0x54, 0xA9, 0x44,
    0x2E, 0xDA, 0x4A, 0xC5, 0xC2, 0x99, 0xF0, 0xDE, 0xCA, 0x77, 0x99, 0x6B,
    0x0C, 0x79, 0xE5, 0x29, 0x74, 0x83, 0x69, 0xEA, 0xB8, 0x72, 0x30, 0x3D,
    0x7A, 0x30, 0xE1, 0x03, 0x7B, 0x09, 0xE6, 0x11, 0xC0, 0xDC, 0xFF, 0xFD,
    0xBD, 0xEC, 0x9C, 0xCC, 0x46, 0x7B, 0x4C, 0x4C, 0x59, 0xBE, 0x82, 0x7C,
    0xF5, 0x60, 0x5A, 0xC3, 0xE8, 0xA8, 0x8A, 0x38, 0x9E, 0x01, 0x57, 0xF1,
    0x79, 0x3A, 0x7C, 0xA3, 0x9F, 0x12, 0x1A, 0x4F, 0x2E, 0xA2, 0xE5, 0x0A,
    0xAB, 0xC0, 0xF4, 0xA5, 0xE3, 0x5F, 0x89, 0x1C, 0x8F, 0xA4, 0x5E, 0xCE,
    0x0D, 0x91, 0x05, 0x1B, 0x17, 0x62, 0x48, 0xFE, 0xA5, 0x4C, 0xEF, 0x2D,
    0x28, 0xF1, 0x5E, 0xE6, 0xD1, 0x30, 0x89, 0x0A, 0xAD, 0x18, 0xAF, 0x6F,
    0x04, 0x09, 0x36, 0x9A, 0xFF, 0xCA, 0xA1, 0xA7, 0x05, 0x7F, 0xD4, 0xBF,
    0x3A, 0xB5, 0x42, 0x6D, 0xE9, 0x07, 0x29, 0x65, 0x8B, 0xAD, 0x4D, 0x0F,
    0x22, 0xE1, 0x59, 0x43, 0x68, 0x87, 0xA8, 0x8B, 0xBC, 0x69, 0xA1, 0x94,
    0x22, 0x3E, 0x8A, 0x49, 0xE8, 0xA3, 0x6F, 0xC2, 0x93, 0x58, 0xE7, 0xAE,
    0xC9, 0x1F, 0xCF, 0x61, 0x93, 0xFC, 0xC1, 0xF6, 0xF3, 0x27, 0x7F, 0x0A,
    0x90, 0xE0, 0x65, 0x32, 0x57, 0x47, 0xE2, 0xED, 0x08, 0x59, 0xA6, 0xF0,
    0x17, 0x2C, 0x13, 0xE0
};

static const unsigned char kat_RSA_SHA1[] = {
    0x3B, 0x60, 0x4B, 0xFC, 0x54, 0x28, 0x23, 0xE6, 0x2F, 0x05, 0x04, 0xBA,
    0x9D, 0xE4, 0x3C, 0xB8, 0x5B, 0x60, 0x5C, 0xCD, 0x9D, 0xEA, 0xC3, 0x4C,
    0xC2, 0x33, 0xE6, 0xC6, 0x21, 0x48, 0x76, 0xEC, 0xB2, 0xF5, 0x11, 0xDE,
    0x44, 0xB4, 0xAF, 0x16, 0x11, 0xC3, 0x18, 0x16, 0xB3, 0x69, 0xBB, 0x94,
    0xED, 0xE8, 0xB3, 0x9E, 0xB1, 0x43, 0x8E, 0xCE, 0xB4, 0x34, 0x9B, 0x08,
    0x22, 0xAF, 0x31, 0x73, 0xB5, 0xFA, 0x11, 0x7E, 0x8F, 0x13, 0x52, 0xEC,
    0xC9, 0x03, 0xEE, 0x0D, 0x2B, 0x91, 0x32, 0xF2, 0x8E, 0xDF, 0x02, 0xE0,
    0x0A, 0x47, 0xD2, 0x0A, 0x51, 0x00, 0x1A, 0x30, 0x6F, 0x0C, 0xB3, 0x54,
    0x64, 0x20, 0x90, 0x0C, 0x01, 0xBE, 0xC0, 0x42, 0x8C, 0x5D, 0x18, 0x6F,
    0x32, 0x75, 0x45, 0x7B, 0x1C, 0x04, 0xA2, 0x9F, 0x84, 0xD7, 0xF5, 0x3A,
    0x95, 0xD4, 0xE8, 0x8D, 0xEC, 0x99, 0xEF, 0x18, 0x5E, 0x64, 0xD3, 0xAF,
    0xF8, 0xD4, 0xFF, 0x3C, 0x87, 0xA0, 0x3F, 0xC7, 0x22, 0x05, 0xFD, 0xFD,
    0x29, 0x8A, 0x28, 0xDA, 0xA9, 0x8A, 0x8B, 0x23, 0x62, 0x9D, 0x42, 0xB8,
    0x4A, 0x76, 0x0D, 0x9F, 0x9A, 0xE0, 0xE6, 0xDD, 0xAD, 0x5E, 0x5F, 0xD5,
    0x32, 0xE9, 0x4B, 0x97, 0x7D, 0x62, 0x0A, 0xB3, 0xBE, 0xF2, 0x8C, 0x1F,
    0x2B, 0x22, 0x06, 0x15, 0x33, 0x71, 0xED, 0x9B, 0xA0, 0x82, 0xCE, 0xBF,
    0x3B, 0x08, 0x5F, 0xA7, 0x20, 0x94, 0x09, 0xEB, 0x82, 0xA5, 0x41, 0x60,
    0xF1, 0x08, 0xEB, 0x8D, 0xCC, 0x8D, 0xC9, 0x52, 0x0A, 0xAF, 0xF4, 0xF9,
    0x9F, 0x82, 0xD8, 0x0B, 0x75, 0x5E, 0xE4, 0xAF, 0x65, 0x96, 0xAF, 0xFC,
    0x33, 0xBF, 0x9F, 0x3E, 0xA4, 0x7B, 0x86, 0xC7, 0xF7, 0x47, 0xAB, 0x37,
    0x05, 0xD6, 0x0D, 0x31, 0x72, 0x8C, 0x80, 0x1E, 0xA9, 0x54, 0xFC, 0xDF,
    0x27, 0x90, 0xE2, 0x01
};

static const unsigned char kat_RSA_SHA224[] = {
    0xA2, 0xD8, 0x42, 0x53, 0xDD, 0xBF, 0x1F, 0x6B, 0x07, 0xE0, 0x60, 0x86,
    0x5A, 0x60, 0x06, 0x8F, 0x44, 0xD9, 0xB0, 0x4A, 0xAA, 0x90, 0x71, 0xB8,
    0xB2, 0xBC, 0x30, 0x41, 0x50, 0xBB, 0xFD, 0x46, 0x98, 0x4D, 0xC0, 0x89,
    0x57, 0x85, 0x8A, 0x97, 0x49, 0x25, 0xA8, 0x0C, 0x69, 0x70, 0x19, 0x39,
    0x66, 0x24, 0xB4, 0x69, 0x47, 0xD2, 0x7C, 0xDE, 0x2D, 0x37, 0x59, 0xB3,
    0xE3, 0xC7, 0x6B, 0xDD, 0xBE, 0xE1, 0xE6, 0x28, 0x9A, 0x8D, 0x42, 0x3E,
    0x28, 0x01, 0xD7, 0x03, 0xC9, 0x73, 0xC3, 0x6B, 0x03, 0xEC, 0x1E, 0xF8,
    0x53, 0x8B, 0x52, 0x42, 0x89, 0x55, 0xB7, 0x87, 0xA9, 0x94, 0xC2, 0xB4,
    0x4B, 0x76, 0xF5, 0x61, 0x47, 0xE1, 0x44, 0x7B, 0xEC, 0xB4, 0x25, 0x66,
    0xC0, 0xFF, 0xEB, 0x86, 0x24, 0xAA, 0xA8, 0x72, 0xC7, 0xFB, 0xFB, 0xF6,
    0x84, 0xA7, 0x5B, 0xD4, 0x87, 0xE5, 0x84, 0x56, 0x1E, 0x4C, 0xE5, 0xBC,
    0x87, 0x94, 0xAC, 0x9C, 0x1B, 0x3D, 0xF7, 0xD4, 0x36, 0x85, 0x9F, 0xC9,
    0xF6, 0x43, 0x3F, 0xB6, 0x25, 0x33, 0x48, 0x0F, 0xE5, 0x7C, 0xCD, 0x53,
    0x48, 0xEB, 0x02, 0x11, 0xB9, 0x9E, 0xC3, 0xB4, 0xE1, 0x54, 0xD6, 0xAA,
    0x1A, 0x9E, 0x10, 0xE1, 0x27, 0x25, 0xF2, 0xE1, 0xAB, 0xAB, 0x6C, 0x45,
    0x61, 0xD5, 0xA3, 0x6C, 0xB6, 0x33, 0x52, 0xAE, 0x3D, 0xFD, 0x22, 0xFC,
    0x3A, 0xAB, 0x63, 0x94, 0xB5, 0x3A, 0x69, 0x11, 0xAC, 0x99, 0x4F, 0x33,
    0x67, 0x0A, 0x1A, 0x70, 0x1E, 0xB9, 0xE2, 0x26, 0x27, 0x68, 0xEA, 0xF5,
    0x97, 0x55, 0xAC, 0x83, 0x6A, 0x40, 0x3B, 0x56, 0xAE, 0x13, 0x88, 0xE8,
    0x98, 0x72, 0x52, 0x91, 0x7F, 0x78, 0x0A, 0x18, 0xD4, 0x44, 0x78, 0x83,
    0x0D, 0x44, 0x77, 0xA6, 0xF3, 0x04, 0xF1, 0x8C, 0xBC, 0x2F, 0xF9, 0x5B,
    0xDB, 0x70, 0x00, 0xF6
};

static const unsigned char kat_RSA_SHA256[] = {
    0xC2, 0xB1, 0x97, 0x00, 0x9A, 0xE5, 0x80, 0x6A, 0xE2, 0x51, 0x68, 0xB9,
    0x7A, 0x0C, 0xF2, 0xB4, 0x77, 0xED, 0x15, 0x0C, 0x4E, 0xE1, 0xDC, 0xFF,
    0x8E, 0xBC, 0xDE, 0xC7, 0x9A, 0x96, 0xF1, 0x47, 0x45, 0x24, 0x9D, 0x6F,
    0xA6, 0xF3, 0x1D, 0x0D, 0x35, 0x4C, 0x1A, 0xF3, 0x58, 0x2C, 0x6C, 0x06,
    0xD6, 0x22, 0x37, 0x77, 0x8C, 0x33, 0xE5, 0x07, 0x53, 0x93, 0x28, 0xCF,
    0x67, 0xFA, 0xC4, 0x1F, 0x1B, 0x24, 0xDB, 0x4C, 0xC5, 0x2A, 0x51, 0xA2,
    0x60, 0x15, 0x8C, 0x54, 0xB4, 0x30, 0xE2, 0x24, 0x47, 0x86, 0xF2, 0xF8,
    0x6C, 0xD6, 0x12, 0x59, 0x2C, 0x74, 0x9A, 0x37, 0xF3, 0xC4, 0xA2, 0xD5,
    0x4E, 0x1F, 0x77, 0xF0, 0x27, 0xCE, 0x77, 0xF8, 0x4A, 0x79, 0x03, 0xBE,
    0xC8, 0x06, 0x2D, 0xA7, 0xA6, 0x46, 0xF5, 0x55, 0x79, 0xD7, 0x5C, 0xC6,
    0x5B, 0xB1, 0x00, 0x4E, 0x7C, 0xD9, 0x11, 0x85, 0xE0, 0xB1, 0x4D, 0x2D,
    0x13, 0xD7, 0xAC, 0xEA, 0x64, 0xD1, 0xAC, 0x8F, 0x8D, 0x8F, 0xEA, 0x42,
    0x7F, 0xF9, 0xB7, 0x7D, 0x2C, 0x68, 0x49, 0x07, 0x7A, 0x74, 0xEF, 0xB4,
    0xC9, 0x97, 0x16, 0x5C, 0x6C, 0x6E, 0x5C, 0x09, 0x2E, 0x8E, 0x13, 0x2E,
    0x1A, 0x8D, 0xA6, 0x0C, 0x6E, 0x0C, 0x1C, 0x0F, 0xCC, 0xB2, 0x78, 0x8A,
    0x07, 0xFC, 0x5C, 0xC2, 0xF5, 0x65, 0xEC, 0xAB, 0x8B, 0x3C, 0xCA, 0x91,
    0x6F, 0x84, 0x7C, 0x21, 0x0E, 0xB8, 0xDA, 0x7B, 0x6C, 0xF7, 0xDF, 0xAB,
    0x7E, 0x15, 0xFD, 0x85, 0x0B, 0x33, 0x9B, 0x6A, 0x3A, 0xC3, 0xEF, 0x65,
    0x04, 0x6E, 0xB2, 0xAC, 0x98, 0xFD, 0xEB, 0x02, 0xF5, 0xC0, 0x0B, 0x5E,
    0xCB, 0xD4, 0x83, 0x82, 0x18, 0x1B, 0xDA, 0xB4, 0xCD, 0xE8, 0x71, 0x6B,
    0x1D, 0xB5, 0x4F, 0xE9, 0xD6, 0x43, 0xA0, 0x0A, 0x14, 0xA0, 0xE7, 0x5D,
    0x47, 0x9D, 0x18, 0xD7
};

static const unsigned char kat_RSA_SHA384[] = {
    0x11, 0x5E, 0x63, 0xFE, 0x47, 0xAA, 0x6A, 0x84, 0xEB, 0x44, 0x9A, 0x00,
    0x96, 0x4A, 0xED, 0xD2, 0xA7, 0x67, 0x3A, 0x64, 0x82, 0x30, 0x61, 0x2D,
    0xE3, 0xF5, 0x49, 0x68, 0x5E, 0x60, 0xD2, 0x4D, 0xEF, 0xF2, 0xA4, 0xB2,
    0x9A, 0x81, 0x1D, 0x41, 0xA5, 0x73, 0x59, 0xEB, 0xBB, 0xC4, 0x9E, 0x2B,
    0xEB, 0xC3, 0xDE, 0x3A, 0xEA, 0xF5, 0xAD, 0xDA, 0x87, 0x08, 0x68, 0xCF,
    0x12, 0x9B, 0xC1, 0xE4, 0xA7, 0x71, 0xF8, 0xBD, 0x6B, 0x6F, 0x50, 0xF1,
    0xD1, 0xFF, 0xCE, 0x6C, 0xD9, 0xBE, 0xDA, 0x76, 0xF3, 0xEB, 0xAB, 0x9C,
    0x41, 0x6E, 0x4F, 0x35, 0x7A, 0x61, 0x27, 0xBC, 0x03, 0x3E, 0xAE, 0x3E,
    0x1B, 0xDD, 0xAC, 0xD9, 0x1A, 0xFF, 0xD3, 0xF5, 0x66, 0x43, 0x07, 0x76,
    0x8A, 0x69, 0x2D, 0x14, 0xB1, 0xBE, 0x55, 0x49, 0x90, 0x89, 0x4B, 0xC4,
    0x11, 0x67, 0xD5, 0x9D, 0xB0, 0xB2, 0xEE, 0x8D, 0x0A, 0x47, 0x4A, 0xD9,
    0x0E, 0xD1, 0x24, 0xF0, 0x30, 0x2B, 0xF2, 0x79, 0x47, 0xDB, 0x70, 0xB4,
    0x46, 0xF2, 0xF8, 0xB7, 0xB4, 0xF6, 0x34, 0x79, 0xA8, 0x2D, 0x3D, 0x56,
    0xD5, 0x9A, 0x60, 0x7A, 0x04, 0xC7, 0x66, 0x1D, 0xCD, 0x3C, 0xD5, 0x39,
    0x37, 0x12, 0x51, 0x5E, 0x9F, 0xF8, 0x1A, 0xAF, 0x13, 0xC1, 0x13, 0x00,
    0x35, 0xD5, 0x8D, 0x17, 0xE3, 0x02, 0x28, 0xD9, 0xEC, 0xDE, 0xD1, 0x2F,
    0x93, 0x49, 0x03, 0x11, 0x3E, 0x56, 0x9D, 0xC2, 0x31, 0xF8, 0xAF, 0x2D,
    0xD9, 0x99, 0xB7, 0x8A, 0xAC, 0x5A, 0x86, 0x20, 0x3A, 0x83, 0x29, 0x26,
    0x9D, 0x03, 0x52, 0x2B, 0x34, 0x56, 0x40, 0x16, 0x53, 0x50, 0x82, 0xC9,
    0xC7, 0xD5, 0x51, 0x4C, 0xED, 0xB3, 0xE2, 0xE1, 0xCF, 0xA8, 0xCE, 0xBD,
    0xB1, 0x48, 0xA6, 0x8A, 0x79, 0x17, 0x55, 0x11, 0xEF, 0xE8, 0x14, 0xF4,
    0x7E, 0x37, 0x1D, 0x96
};

static const unsigned char kat_RSA_SHA512[] = {
    0x35, 0x6D, 0xF1, 0x9E, 0xCF, 0xB1, 0xF6, 0x0C, 0x04, 0x21, 0x17, 0xB3,
    0xC4, 0x9D, 0xFE, 0x62, 0x1C, 0x1A, 0x45, 0x00, 0x2E, 0x6B, 0xB6, 0x9F,
    0x5C, 0xB1, 0xCB, 0xCF, 0xF9, 0x67, 0xEA, 0x62, 0x8A, 0xEB, 0x77, 0x02,
    0x42, 0x30, 0x88, 0xB1, 0x48, 0xDF, 0x12, 0x60, 0x6E, 0x92, 0xBB, 0x4B,
    0x09, 0x68, 0xD1, 0x70, 0x2B, 0x59, 0xEE, 0x57, 0x96, 0xF9, 0xEA, 0xA3,
    0x4C, 0xE9, 0xC9, 0xBD, 0x25, 0x34, 0x66, 0x15, 0x6C, 0xC9, 0x81, 0xD1,
    0x48, 0x0F, 0x33, 0x5F, 0x05, 0x4F, 0xC2, 0xC4, 0xDD, 0x09, 0x54, 0x79,
    0xA1, 0x57, 0x07, 0x70, 0xA0, 0x33, 0x02, 0x4D, 0x5D, 0xE9, 0x24, 0xD1,
    0xEF, 0xF0, 0x61, 0xD0, 0x1D, 0x41, 0xE2, 0x9B, 0x2B, 0x7C, 0xD0, 0x4E,
    0x55, 0xD9, 0x6D, 0xA1, 0x16, 0x9F, 0xDA, 0xC3, 0x3B, 0xF1, 0x74, 0xD1,
    0x99, 0xF1, 0x63, 0x57, 0xAD, 0xC7, 0x55, 0xF4, 0x97, 0x43, 0x1C, 0xED,
    0x1B, 0x7A, 0x32, 0xCB, 0x24, 0xA6, 0x3D, 0x93, 0x37, 0x90, 0x74, 0xEE,
    0xD2, 0x8D, 0x4B, 0xBC, 0x72, 0xDA, 0x25, 0x2B, 0x64, 0xE9, 0xCA, 0x69,
    0x36, 0xB6, 0xEC, 0x6E, 0x8F, 0x33, 0x0E, 0x74, 0x40, 0x48, 0x51, 0xE2,
    0x54, 0x6F, 0xAF, 0x6E, 0x36, 0x54, 0x3A, 0xEC, 0x78, 0x37, 0xE6, 0x1F,
    0x76, 0xA5, 0x4D, 0xA6, 0xD9, 0xB3, 0x6B, 0x17, 0x6D, 0x61, 0xFC, 0xA3,
    0x85, 0x4A, 0xCC, 0xDA, 0x52, 0xAC, 0x5B, 0xDA, 0x51, 0xE5, 0x7F, 0x5B,
    0x52, 0x8B, 0x74, 0x75, 0x99, 0x5C, 0x01, 0xFD, 0x25, 0x3E, 0xCD, 0x86,
    0x6F, 0x7A, 0xC0, 0xD8, 0x17, 0x6F, 0xD1, 0xD2, 0x6B, 0xAB, 0x14, 0x1F,
    0x3B, 0xB8, 0x15, 0x05, 0x86, 0x40, 0x36, 0xCF, 0xDA, 0x59, 0x2B, 0x9A,
    0xE9, 0x1E, 0x6E, 0xD3, 0x6B, 0xA1, 0x19, 0xC5, 0xE6, 0x3F, 0xE9, 0x2E,
    0x43, 0xA8, 0x34, 0x0A
};

static int fips_rsa_encrypt_test(RSA *rsa, const unsigned char *plaintext,
                                 int ptlen)
{
    unsigned char *ctbuf = NULL, *ptbuf = NULL;
    int ret = 0;
    int len;

    ctbuf = OPENSSL_malloc(RSA_size(rsa));
    if (!ctbuf)
        goto err;

    len = RSA_public_encrypt(ptlen, plaintext, ctbuf, rsa, RSA_PKCS1_PADDING);
    if (len <= 0)
        goto err;
    /* Check ciphertext doesn't match plaintext */
    if (len >= ptlen && !memcmp(plaintext, ctbuf, ptlen))
        goto err;

    ptbuf = OPENSSL_malloc(RSA_size(rsa));
    if (!ptbuf)
        goto err;

    len = RSA_private_decrypt(len, ctbuf, ptbuf, rsa, RSA_PKCS1_PADDING);
    if (len != ptlen)
        goto err;
    if (memcmp(ptbuf, plaintext, len))
        goto err;

    ret = 1;

 err:
    if (ctbuf)
        OPENSSL_free(ctbuf);
    if (ptbuf)
        OPENSSL_free(ptbuf);
    return ret;
}

int FIPS_selftest_rsa()
{
    int ret = 0;
    RSA *key;
    EVP_PKEY *pk = NULL;

    if ((key = RSA_new()) == NULL)
        goto err;
    setrsakey(key);
    if ((pk = EVP_PKEY_new()) == NULL)
        goto err;

    EVP_PKEY_set1_RSA(pk, key);

    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_SHA1, sizeof(kat_RSA_SHA1),
                                  EVP_sha1(), EVP_MD_CTX_FLAG_PAD_PKCS1,
                                  "RSA SHA1 PKCS#1"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_SHA224, sizeof(kat_RSA_SHA224),
                                  EVP_sha224(), EVP_MD_CTX_FLAG_PAD_PKCS1,
                                  "RSA SHA224 PKCS#1"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_SHA256, sizeof(kat_RSA_SHA256),
                                  EVP_sha256(), EVP_MD_CTX_FLAG_PAD_PKCS1,
                                  "RSA SHA256 PKCS#1"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_SHA384, sizeof(kat_RSA_SHA384),
                                  EVP_sha384(), EVP_MD_CTX_FLAG_PAD_PKCS1,
                                  "RSA SHA384 PKCS#1"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_SHA512, sizeof(kat_RSA_SHA512),
                                  EVP_sha512(), EVP_MD_CTX_FLAG_PAD_PKCS1,
                                  "RSA SHA512 PKCS#1"))
        goto err;

    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_PSS_SHA1, sizeof(kat_RSA_PSS_SHA1),
                                  EVP_sha1(), EVP_MD_CTX_FLAG_PAD_PSS,
                                  "RSA SHA1 PSS"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_PSS_SHA224,
                                  sizeof(kat_RSA_PSS_SHA224), EVP_sha224(),
                                  EVP_MD_CTX_FLAG_PAD_PSS, "RSA SHA224 PSS"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_PSS_SHA256,
                                  sizeof(kat_RSA_PSS_SHA256), EVP_sha256(),
                                  EVP_MD_CTX_FLAG_PAD_PSS, "RSA SHA256 PSS"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_PSS_SHA384,
                                  sizeof(kat_RSA_PSS_SHA384), EVP_sha384(),
                                  EVP_MD_CTX_FLAG_PAD_PSS, "RSA SHA384 PSS"))
        goto err;
    if (!fips_pkey_signature_test(pk, kat_tbs, sizeof(kat_tbs) - 1,
                                  kat_RSA_PSS_SHA512,
                                  sizeof(kat_RSA_PSS_SHA512), EVP_sha512(),
                                  EVP_MD_CTX_FLAG_PAD_PSS, "RSA SHA512 PSS"))
        goto err;

    if (!fips_rsa_encrypt_test(key, kat_tbs, sizeof(kat_tbs) - 1))
        goto err;

    ret = 1;

 err:
    if (pk)
        EVP_PKEY_free(pk);
    if (key)
        RSA_free(key);
    return ret;
}

#endif                          /* def OPENSSL_FIPS */
