#!/usr/bin/python
# @lint-avoid-python-3-compatibility-imports
#
# aliext4writeslower  Trace slow ext4 write operation.
#             For Linux, uses BCC, eBPF.
#
# USAGE: aliext4writeslower [-h] [-p PID] [-d device] [min_ms]
#
# This script traces key steps of write(2) which maybe time-consuming,
# here trace 4 key steps: jbd2 handle start(it may need jbd2 trancation
# checkpoint or commit), page cache allocation, block allocation and
# balance dirty pages, and we also take into account sched delay.
#
# With this tool, we can clearly know write(2)'s latency proportion.
#
# WARNING: This adds low-overhead instrumentation to these ext4 operations,
# including reads and writes from the file system cache. Such reads and writes
# can be very frequent (depending on the workload; eg, 1M/sec), at which
# point the overhead of this tool (even if it prints no "slower" events) can
# begin to become significant.
#
# By default, a minimum millisecond threshold of 10 is used.
#
# Licensed under the Apache License, Version 2.0 (the "License")
#
# 2019-08-29   Xiaoguang Wang   Created this.

from __future__ import print_function
from bcc import BPF
import argparse
from time import strftime
import ctypes as ct

# symbols
kallsyms = "/proc/kallsyms"

# arguments
examples = """examples:
    ./aliext4writeslower             # trace write(2) slower than 10 ms (default)
    ./aliext4writeslower 1           # trace write(2) slower than 1 ms
    ./aliext4writeslower 0           # trace all operations (warning: verbose)
    ./aliext4writeslower -d sda5     # inspect specified device /dev/sda5 containing ext4 fs
    ./aliext4writeslower -p 185      # trace PID 185 only
"""
parser = argparse.ArgumentParser(
    description="Trace common ext4 write(2) slower than a threshold",
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=examples)
parser.add_argument("-p", "--pid", help="trace this PID only")
parser.add_argument("-d", "--device", help="inspect specified device")
parser.add_argument("min_ms", nargs="?", default='10',
    help="minimum I/O duration to trace, in ms (default 10)")
parser.add_argument("--ebpf", action="store_true",
    help=argparse.SUPPRESS)
args = parser.parse_args()
min_ms = int(args.min_ms)
pid = args.pid
debug = 0

# define BPF program
bpf_text = """
#include <uapi/linux/ptrace.h>
#include <linux/sched.h>
#include <linux/blkdev.h>

enum {
	TRACE_PAGECACHE_ALLOC = 0,
	TRACE_JOURNAL_START,
	TRACE_BLOCK_ALLOC,
	TRACE_BALANCE_DIRTY,
	TRACE_WRITE_BEGIN,
	TRACE_WRITE_END,
	TRACE_TYPE_MAX,
};

struct val_t {
	dev_t dev;
	u64 ts;
	u64 sched_delay;
	u64 trace_ts[TRACE_TYPE_MAX];
	u64 trace_time[TRACE_TYPE_MAX];
	u64 offset;
	struct file *fp;
};

struct data_t {
	dev_t dev;
	u64 size;
	u64 offset;
	u64 delta_us;
	u64 pagecache_alloc_us;
	u64 journal_start_us;
	u64 block_alloc_us;
	u64 balance_dirty_us;
	u64 write_begin_us;
	u64 write_end_us;
	u64 sched_delay;
	u64 pid;
	char task[TASK_COMM_LEN];
	char file[DNAME_INLINE_LEN];
};

BPF_HASH(entryinfo, u64, struct val_t);
BPF_PERF_OUTPUT(events);


// ext4_file_write_iter():
int trace_write_entry(struct pt_regs *ctx, struct kiocb *iocb)
{
	struct val_t val = {};
	u64 id = bpf_get_current_pid_tgid();
	u32 pid = id >> 32; // PID is higher part
	struct task_struct *p = (struct task_struct *)bpf_get_current_task();
	struct inode *inode = iocb->ki_filp->f_inode;
	dev_t dev = inode->i_sb->s_bdev->bd_dev;

	if (FILTER_DEV)
		return 0;

	if (FILTER_PID)
		return 0;

	val.ts = bpf_ktime_get_ns();
	val.sched_delay = p->sched_info.run_delay;
	val.fp = iocb->ki_filp;
	val.dev = dev;
	val.offset = iocb->ki_pos;
	if (val.fp)
		entryinfo.update(&id, &val);

	return 0;
}

int trace_write_return(struct pt_regs *ctx)
{
	struct val_t *valp;
	u64 ts, delta_us, id = bpf_get_current_pid_tgid();
	u32 size, pid = id >> 32;
	struct task_struct *p = (struct task_struct *)bpf_get_current_task();
	struct data_t data = {};

	if (FILTER_PID)
		return 0;

	valp = entryinfo.lookup(&id);
	if (valp == 0)
		return 0;

	ts = bpf_ktime_get_ns();
	delta_us = (ts - valp->ts) / 1000;
	entryinfo.delete(&id);
	if (FILTER_US)
		return 0;

	size = PT_REGS_RC(ctx);
	data.size = size;
	data.delta_us = delta_us;
	data.pid = pid;
	data.sched_delay = (p->sched_info.run_delay - valp->sched_delay) / 1000;
	data.offset = valp->offset;
	data.dev = valp->dev;
	data.pagecache_alloc_us = valp->trace_time[TRACE_PAGECACHE_ALLOC] / 1000;
	data.journal_start_us = valp->trace_time[TRACE_JOURNAL_START] / 1000;
	data.block_alloc_us = valp->trace_time[TRACE_BLOCK_ALLOC] / 1000;
	data.balance_dirty_us = valp->trace_time[TRACE_BALANCE_DIRTY] / 1000;
	data.write_begin_us = valp->trace_time[TRACE_WRITE_BEGIN] / 1000;
	data.write_end_us = valp->trace_time[TRACE_WRITE_END] / 1000;
	bpf_get_current_comm(&data.task, sizeof(data.task));

	// workaround (rewriter should handle file to d_name in one step):
	struct dentry *de = NULL;
	struct qstr qs = {};
	bpf_probe_read(&de, sizeof(de), &valp->fp->f_path.dentry);
	bpf_probe_read(&qs, sizeof(qs), (void *)&de->d_name);
	if (qs.len == 0)
		return 0;
	bpf_probe_read(&data.file, sizeof(data.file), (void *)qs.name);

	// output
	events.perf_submit(ctx, &data, sizeof(data));

	return 0;
}


static int trace_type_entry(struct pt_regs *ctx, int type)
{
    struct val_t *valp;
    u64 id = bpf_get_current_pid_tgid();
    u32 pid = id >> 32; // PID is higher part
    u64 ts = bpf_ktime_get_ns();

    valp = entryinfo.lookup(&id);
    if (valp == 0) {
        // missed tracing issue or filtered
        return 0;
    }

    valp->trace_ts[type] = ts;
    return 0;
}

static int trace_type_return(struct pt_regs *ctx, int type)
{
    struct val_t *valp;
    u64 id = bpf_get_current_pid_tgid();
    u32 pid = id >> 32; // PID is higher part
    u64 ts = bpf_ktime_get_ns();

    valp = entryinfo.lookup(&id);
    if (valp == 0) {
        // missed tracing issue or filtered
        return 0;
    }

    valp->trace_time[type] += ts - valp->trace_ts[type];;
    return 0;
}

int trace_pagecache_alloc_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_PAGECACHE_ALLOC);
}

int trace_pagecache_alloc_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_PAGECACHE_ALLOC);
}

int trace_journal_start_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_JOURNAL_START);
}

int trace_journal_start_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_JOURNAL_START);
}

int trace_block_alloc_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_BLOCK_ALLOC);
}

int trace_block_alloc_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_BLOCK_ALLOC);
}

int trace_balance_dirty_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_BALANCE_DIRTY);
}

int trace_balance_dirty_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_BALANCE_DIRTY);
}

int trace_aops_write_begin_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_WRITE_BEGIN);
}

int trace_aops_write_begin_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_WRITE_BEGIN);
}

int trace_aops_write_end_entry(struct pt_regs *ctx)
{
	return trace_type_entry(ctx, TRACE_WRITE_END);
}

int trace_aops_write_end_return(struct pt_regs *ctx)
{
	return trace_type_return(ctx, TRACE_WRITE_END);
}
"""

devid2name={}
num_trans=0

def init_dev_name():
        global devid2name

        f = open("/proc/partitions")
        for line in f.readlines():
                line = line.strip()
                if not len(line):
                        continue;

                str = line.split()
                if str[0].isalpha():
                        continue
                dev_num = (int(str[0]) << 20) + int(str[1])
                devid2name[dev_num] = str[3]
        f.close()

init_dev_name()

name2devid={v:k for k,v in devid2name.items()}

if args.device:
    bpf_text = bpf_text.replace('FILTER_DEV', 'dev != %u' % name2devid[args.device])
else:
    bpf_text = bpf_text.replace('FILTER_DEV', '0')

if min_ms == 0:
    bpf_text = bpf_text.replace('FILTER_US', '0')
else:
    bpf_text = bpf_text.replace('FILTER_US', 'delta_us <= %s' % str(min_ms * 1000))

if args.pid:
    bpf_text = bpf_text.replace('FILTER_PID', 'pid != %s' % pid)
else:
    bpf_text = bpf_text.replace('FILTER_PID', '0')

if debug or args.ebpf:
    print(bpf_text)
    if args.ebpf:
        exit()

# kernel->user event data: struct data_t
DNAME_INLINE_LEN = 32   # linux/dcache.h
TASK_COMM_LEN = 16      # linux/sched.h
class Data(ct.Structure):
    _fields_ = [
        ("dev", ct.c_uint),
        ("size", ct.c_ulonglong),
        ("offset", ct.c_ulonglong),
        ("delta_us", ct.c_ulonglong),
        ("pagecache_alloc_us", ct.c_ulonglong),
        ("journal_start_us", ct.c_ulonglong),
        ("block_alloc_us", ct.c_ulonglong),
        ("balance_dirty_us", ct.c_ulonglong),
        ("write_begin_us", ct.c_ulonglong),
        ("write_end_us", ct.c_ulonglong),
        ("sched_delay", ct.c_ulonglong),
        ("pid", ct.c_ulonglong),
        ("task", ct.c_char * TASK_COMM_LEN),
        ("file", ct.c_char * DNAME_INLINE_LEN)
    ]

# process event
def print_event(cpu, data, size):
    event = ct.cast(data, ct.POINTER(Data)).contents

    print("%-8s %-8s %-14.14s %-6s %-7s %-8d %7.2f %8.2f %10.2f %11.2f %13.2f %11.2f %9.2f %11.2f %s" %
	(strftime("%H:%M:%S"), devid2name[event.dev],
        event.task.decode(), event.pid, event.size, event.offset / 1024,
        float(event.delta_us) / 1000, float(event.pagecache_alloc_us) / 1000,
	float(event.journal_start_us) / 1000, float(event.block_alloc_us) / 1000,
	float(event.balance_dirty_us) / 1000, float(event.write_begin_us) / 1000,
	float(event.write_end_us) / 1000, float(event.sched_delay) / 1000,
	event.file.decode()))

# initialize BPF
b = BPF(text=bpf_text)

b.attach_kprobe(event="ext4_file_write_iter", fn_name="trace_write_entry")
b.attach_kretprobe(event="ext4_file_write_iter", fn_name="trace_write_return")

# trace address_space_operations .write_begin()
b.attach_kprobe(event="ext4_da_write_begin", fn_name="trace_aops_write_begin_entry")
b.attach_kprobe(event="ext4_write_begin", fn_name="trace_aops_write_begin_entry")
b.attach_kretprobe(event="ext4_da_write_begin", fn_name="trace_aops_write_begin_return")
b.attach_kretprobe(event="ext4_write_begin", fn_name="trace_aops_write_begin_return")

# trace address_space_operations .write_end()
b.attach_kprobe(event="ext4_da_write_end", fn_name="trace_aops_write_end_entry")
b.attach_kprobe(event="ext4_write_end", fn_name="trace_aops_write_end_entry")
b.attach_kretprobe(event="ext4_da_write_end", fn_name="trace_aops_write_end_return")
b.attach_kretprobe(event="ext4_write_end", fn_name="trace_aops_write_end_return")

# trace page cache allocation
b.attach_kprobe(event="grab_cache_page_write_begin", fn_name="trace_pagecache_alloc_entry")
b.attach_kretprobe(event="grab_cache_page_write_begin", fn_name="trace_pagecache_alloc_return")

# trace journal start
b.attach_kprobe(event="jbd2__journal_start", fn_name="trace_journal_start_entry")
b.attach_kretprobe(event="jbd2__journal_start", fn_name="trace_journal_start_return")

# trace block allocation
b.attach_kprobe(event="ext4_da_get_block_prep", fn_name="trace_block_alloc_entry")
b.attach_kretprobe(event="ext4_da_get_block_prep", fn_name="trace_block_alloc_return")
b.attach_kprobe(event="_ext4_get_block", fn_name="trace_block_alloc_entry")
b.attach_kretprobe(event="_ext4_get_block", fn_name="trace_block_alloc_return")

# trace balance dirty pages
b.attach_kprobe(event="balance_dirty_pages_ratelimited", fn_name="trace_balance_dirty_entry")
b.attach_kretprobe(event="balance_dirty_pages_ratelimited", fn_name="trace_balance_dirty_return")

# header
if min_ms == 0:
    print("Tracing ext4 operations")
else:
    print("Tracing ext4 operations slower than %d ms" % min_ms)

print("%-8s %-8s %-14s %-6s %-7s %-8s %7s %8s %10s %11s %13s %11s %9s %11s %s" % ("TIME",
      "DEVICE", "COMM", "PID", "BYTES", "OFF_KB", "LAT(ms)", "pg_alloc", "jbd2_start",
      "block_alloc", "balance_dirty", "write_begin", "write_end", "sched_delay", "FILENAME"))

# read events
b["events"].open_perf_buffer(print_event, page_cnt=64)
while 1:
    try:
        b.perf_buffer_poll()

    except KeyboardInterrupt:
        exit()
