# kernel selection spoke classes
#
# Copyright (C) 2021 OpenAnolis Community
#
import re
import gi
gi.require_version("Gtk", "3.0")
gi.require_version("Pango", "1.0")

from gi.repository import Gtk, Pango

from pyanaconda.flags import flags
from pyanaconda.core.i18n import _, C_, CN_
from pyanaconda.threading import threadMgr, AnacondaThread
from pyanaconda.payload.manager import payloadMgr, PayloadState
from pyanaconda.payload.base import Payload
from pyanaconda.payload.errors import NoSuchGroup, PayloadError, DependencyError

from pyanaconda.core import util, constants

from pyanaconda.ui.communication import hubQ
from pyanaconda.ui.gui.spokes import NormalSpoke
from pyanaconda.ui.gui.spokes.lib.detailederror import DetailedErrorDialog
from pyanaconda.ui.gui.utils import blockedHandler, escape_markup
from pyanaconda.core.async_utils import async_action_wait
from pyanaconda.ui.categories.software import SoftwareCategory

from pyanaconda.anaconda_loggers import get_module_logger
log = get_module_logger(__name__)

import sys, copy

__all__ = ["KernelSelectionSpoke"]

class KernelSelectionSpoke(NormalSpoke):
    builderObjects = ["kernelWindow"]
    mainWidgetName = "kernelWindow"
    uiFile = "spokes/kernel_selection.glade"
    help_id = "KernelSelectionSpoke"

    category = SoftwareCategory

    #icon = "applications-system"
    icon = "package-x-generic-symbolic"
    title = CN_("GUI|Spoke", "_Kernel Selection")

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._error_msgs = None
        self._orig_kernel = None
        self._kickstarted = flags.automatedInstall and self.data.packages.seen

        self._kernelListBox = self.builder.get_object("kernelListBox")
        self._kernelViewport = self.builder.get_object("kernelViewport")
        self._kernelListBox.set_focus_vadjustment(Gtk.Scrollable.get_vadjustment(self._kernelViewport))

        self._fakeRadio = Gtk.RadioButton(group=None)
        self._fakeRadio.set_active(True)


    @property
    def available_kernels(self):
        return self.payload.available_kernels

    @property
    def current_kernel(self):
        return self.payload.current_kernel


    @current_kernel.setter
    def current_kernel(self, value):
        self.payload.current_kernel = value

    def apply(self):
        self._apply()

    def _apply(self):
        hubQ.send_not_ready(self.__class__.__name__)
        hubQ.send_not_ready("SourceSpoke")
        threadMgr.add(AnacondaThread(name="AnaCheckKernel", target=self.checkKernelSelection))

    def checkKernelSelection(self):
        # we do this only kernel changed
        if self.changed:
            kernel_keys = self.available_kernels.keys()
            for kernel in kernel_keys:
                if kernel == self.current_kernel:
                    self.payload.data.packages.packageList.extend(self.available_kernels[kernel])
                    for package in self.available_kernels[kernel]:
                        if package in self.payload.data.packages.excludedList:
                            self.payload.data.packages.excludedList.remove(package)
                else:
                    self.payload.data.packages.excludedList.extend(self.available_kernels[kernel])
                    for package in self.available_kernels[kernel]:
                        if package in self.payload.data.packages.packageList:
                            self.payload.data.packages.packageList.remove(package)
            # store kernel selection
            self._orig_kernel = self.current_kernel

        hubQ.send_ready(self.__class__.__name__, False)
        hubQ.send_ready("SourceSpoke", False)

    def initialize(self):
        super().initialize()
        self.initialize_start()
        threadMgr.add(AnacondaThread(name="AnaKernelWatcher", target=self._initialize))

    def _initialize(self):
        threadMgr.wait(constants.THREAD_PAYLOAD)

        self.payload.detectMultiKernel()

        # find kernel in package list
        kernel_pattern=re.compile(r"kernel-[4,5].\d+.\d+")
        if self._kickstarted:
            for package in self.payload.data.packages.packageList:
                if kernel_pattern.search(package):
                    self.current_kernel = package.split('-')[1]

        # use first kernel as default, normally it's rhck
        if not self.current_kernel:
            # Check if self.available_kernels is empty
            # When there are no RPMs repo in DVD iso, and net repo is not set, the available_kernels will be empty.
            if len(self.available_kernels) > 0:
                self.current_kernel = list(self.available_kernels.keys())[0]

        hubQ.send_ready(self.__class__.__name__, False)
        self._apply()
        # report that software spoke initialization has been completed
        self.initialize_done()

    @property
    def completed(self):
        processingDone = bool(not threadMgr.get("AnaCheckKernel") and
                              not threadMgr.get(constants.THREAD_PAYLOAD) and
                              not self._error_msgs)

        if processingDone:
            return True

    @property
    def changed(self):
        if self.current_kernel == self._orig_kernel:
            return False
        else:
            return True

    @property
    def mandatory(self):
        return True

    @property
    def ready(self):
        return bool(not threadMgr.get("AnaKernelWatcher") and
                    not threadMgr.get(constants.THREAD_PAYLOAD) and
                    not threadMgr.get("AnaCheckKernel")
                   )

    @property
    def showable(self):
        return isinstance(self.payload, Payload)

    @property
    def status(self):
        if self._error_msgs:
            return _("Error select kernel")
        return self.current_kernel


    def _add_row(self, listbox, name, desc, button, clicked):
        row = Gtk.ListBoxRow()
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)

        button.set_valign(Gtk.Align.START)
        button.connect("toggled", clicked, row)
        box.add(button)

        label = Gtk.Label(label="<b>%s</b>\n%s" % (escape_markup(name), escape_markup(desc)),
                          use_markup=True, wrap=True, wrap_mode=Pango.WrapMode.WORD_CHAR,
                          hexpand=True, xalign=0, yalign=0.5)
        box.add(label)

        row.add(box)
        listbox.insert(row, -1)

    def refresh(self):
        super().refresh()

        threadMgr.wait(constants.THREAD_PAYLOAD)

        self._clear_listbox(self._kernelListBox)
        # Obtain available_kernels by payload, after refresh the RPMs repo by 'Installation Source' manually.  
        log.info('To trigger payload.detectMultiKernel')
        self.payload.detectMultiKernel()
        # use first kernel as default, normally it's rhck
        if not self.current_kernel:
            # Check if self.available_kernels is empty
            # When there are no RPMs repo in DVD iso, and net repo is not set, the available_kernels will be empty.
            if len(self.available_kernels) > 0:
                log.info('len of available_kernels: %d', len(self.available_kernels))
                self.current_kernel = list(self.available_kernels.keys())[0]

        hubQ.send_ready(self.__class__.__name__, False)
        self._apply()

        kernel_keys = self.available_kernels.keys()
        for kernel in kernel_keys:
            radio = Gtk.RadioButton(group=self._fakeRadio)
            if not self.current_kernel:
                self.current_kernel = kernel
            radio.set_active(self.current_kernel and kernel == self.current_kernel)
            if kernel == '4.18.0':
                kernel_desc = _("RHCK")
                kernel_info = _("Compatible with RHEL")
            elif kernel == '4.19.91':
                kernel_desc = _("ANCK")
                kernel_info = _("Support Anolis OS verified platform")
            else:
                kernel_desc = ''
                kernel_info = ''
            self._add_row(self._kernelListBox, "%s (%s)" % (kernel, kernel_desc), "%s (%s)" % (kernel_info, self.available_kernels[kernel][0]), radio, self.on_radio_button_toggled)

        self._kernelListBox.show_all()

    def _clear_listbox(self, listbox):
        for child in listbox.get_children():
            listbox.remove(child)
            del(child)

    # Signal handlers
    def on_radio_button_toggled(self, radio, row):
        # If the radio button toggled to inactive, don't reactivate the row
        if not radio.get_active():
            return
        row.activate()

    def on_kernel_activated(self, listbox, row):
        box = row.get_children()[0]
        button = box.get_children()[0]

        with blockedHandler(button, self.on_radio_button_toggled):
            button.set_active(True)

        kernel_keys = list(self.available_kernels.keys())
        self.current_kernel=kernel_keys[row.get_index()]
