// SPDX-License-Identifier: Apache-2.0

#ifndef _LIBNMSTATE_H_
#define _LIBNMSTATE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

#define NMSTATE_VERSION_MAJOR        2
#define NMSTATE_VERSION_MINOR        2
#define NMSTATE_VERSION_MICRO        35

#define NMSTATE_VERSION              \
    ((NMSTATE_VERSION_MAJOR * 10000) + \
     (NMSTATE_VERSION_MINOR * 100) + \
     NMSTATE_VERSION_MICRO)

#define NMSTATE_PASS                 0
#define NMSTATE_FAIL                 1

#define NMSTATE_FLAG_NONE                   0
#define NMSTATE_FLAG_KERNEL_ONLY            1 << 1
#define NMSTATE_FLAG_NO_VERIFY              1 << 2
#define NMSTATE_FLAG_INCLUDE_STATUS_DATA    1 << 3
#define NMSTATE_FLAG_INCLUDE_SECRETS        1 << 4
#define NMSTATE_FLAG_NO_COMMIT              1 << 5
#define NMSTATE_FLAG_MEMORY_ONLY            1 << 6
#define NMSTATE_FLAG_RUNNING_CONFIG_ONLY    1 << 7
#define NMSTATE_FLAG_YAML_OUTPUT            1 << 8

/**
 * nmstate_net_state_retrieve - Retrieve network state
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Retrieve network state in the format of JSON or YAML.
 *
 * @flags:
 *      Flags for special use cases:
 *          * NMSTATE_FLAG_NONE
 *              No flag
 *          * NMSTATE_FLAG_KERNEL_ONLY
 *              Do not use external plugins, show kernel status only.
 *          * NMSTATE_FLAG_INCLUDE_SECRETS
 *              No not hide sercerts like password.
 *          * NMSTATE_FLAG_RUNNING_CONFIG_ONLY
 *              Only include running config excluding running status like auto
 *              IP addresses and routes, LLDP neighbors.
 *          * NMSTATE_FLAG_YAML_OUTPUT
 *              Show the state in YAML format
 * @state:
 *      Output pointer of char array for network state in json format.
 *      The memory should be freed by nmstate_net_state_free().
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_retrieve(uint32_t flags, char **state, char **log,
                               char **err_kind, char **err_msg);

/**
 * nmstate_net_state_apply - Apply network state
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Apply network state in the format of JSON or YAML.
 *
 * @flags:
 *      Flags for special use cases:
 *          * NMSTATE_FLAG_NONE
 *              No flag
 *          * NMSTATE_FLAG_KERNEL_ONLY
 *              Do not use external plugins, apply to kernel only.
 *          * NMSTATE_FLAG_NO_VERIFY
 *              Do not verify state after applied
 *          * NMSTATE_FLAG_NO_COMMIT
 *              Do not commit new state after verification
 *          * NMSTATE_FLAG_MEMORY_ONLY
 *              No not store network state to persistent.
 * @state:
 *      Pointer of char array for network state in json format.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_apply(uint32_t flags, const char *state,
                            uint32_t rollback_timeout, char **log,
                            char **err_kind, char **err_msg);

/**
 * nmstate_checkpoint_commit - Destroy the checkpoint
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Destroy the checkpoint, if no checkpoint
 *      is passed it will destroy last active checkpoint
 *
 * @checkpoint:
 *      Checkpoint to destroy or empty to select the last active one.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_checkpoint_commit(const char *checkpoint, char **log,
                              char **err_kind, char **err_msg);

/**
 * nmstate_checkpoint_rollback - Rollback the checkpoint
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Rollack the checkpoint, if no checkpoint
 *      is passed it will rollback last active checkpoint
 *
 * @checkpoint:
 *      Checkpoint to rollback or empty to select the last active one.
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_checkpoint_rollback(const char *checkpoint, char **log,
                                char **err_kind, char **err_msg);

/**
 * nmstate_generate_configurations - Generate network configurations
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Generate offline configrations of each backend based on specified
 *      network state. The returned configs is json string for
 *      HashMap/Dictionary with backend name as key, and array of string
 *      as value.
 *
 * @state:
 *      Pointer of char array for network state in JSON or YAML format.
 * @configs:
 *      Output pointer of char array for network configures in JSON or
 *      YAML(depend on which format you use in @state) format.
 *      The memory should be freed by nmstate_net_state_free().
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_generate_configurations(const char *state, char **configs,
                                    char **log, char **err_kind,
                                    char **err_msg);

/**
 * nmstate_net_state_from_policy - Generate network state from policy
 *
 * Version:
 *      2.2
 *
 * Description:
 *      Generate new network state from policy again specified state
 *
 * @policy:
 *      Pointer of char array for network policy in JSON/YAML format.
 * @current_state:
 *      Pointer of char array for current network state.
 * @state:
 *      Output pointer of char array for network state in JSON/YAML format.
 *      The memory should be freed by nmstate_net_state_free().
 * @log:
 *      Output pointer of char array for logging.
 *      The memory should be freed by nmstate_log_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_from_policy(const char *policy,
                                  const char *current_state,
                                  char **state,
                                  char **log,
                                  char **err_kind,
                                  char **err_msg);
/**
 * nmstate_cstring_free - free the memory of C string
 *
 * Version:
 *      0.1
 *
 * Description:
 *      Free the memory of C string.
 *
 * @cstring:
 *      Pointer of char array for string
 *
 * Return:
 *      void
 */
void nmstate_cstring_free(char *cstring);

/**
 * nmstate_generate_differences - Generate network differences
 *
 * Version:
 *      2.2.31
 *
 * Description:
 *      Given new network state and old network state in JSON or YAML
 *      format, output new state with differences only.
 *      The returned configs is JSON string or YAML string matching the format
 *      of new state.
 *
 * @new_state:
 *      Pointer of char array for new network state in JSON or YAML format.
 * @old_state:
 *      Pointer of char array for old network state in JSON or YAML format.
 * @diff_state:
 *      Output pointer of char array for new network state with differences
 *      only in JSON or YAML(depend on which format you use in @state) format.
 *      The memory should be freed by nmstate_net_state_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_generate_differences(const char *new_state,
                                 const char *old_state,
                                 char **diff_state,
                                 char **err_kind,
                                 char **err_msg);

/**
 * nmstate_net_state_format - Tidy up network state in YAML/JSON format
 *
 * Version:
 *      2.2.32
 *
 * Description:
 *      Given network state in JSON or YAML format, output state JSON/YAML
 *      with better layout and item ordering.
 *      The returned configs is JSON string or YAML string matching the format
 *      of input state.
 *
 * @state:
 *      Pointer of char array for network state in JSON or YAML format.
 * @formated_state:
 *      Output pointer of char array for formated network state
 *      in JSON or YAML(depend on which format you use in @state) format.
 *      The memory should be freed by nmstate_net_state_free().
 * @err_kind:
 *      Output pointer of char array for error kind.
 *      The memory should be freed by nmstate_err_kind_free().
 * @err_msg:
 *      Output pointer of char array for error message.
 *      The memory should be freed by nmstate_err_msg_free().
 *
 * Return:
 *      Error code:
 *          * NMSTATE_PASS
 *              On success.
 *          * NMSTATE_FAIL
 *              On failure.
 */
int nmstate_net_state_format(const char *state,
                             char **formated_state,
                             char **err_kind,
                             char **err_msg);



#endif // _LIBNMSTATE_H_
