use base64::engine::general_purpose::URL_SAFE_NO_PAD;
use base64::Engine as _;
use flatten_json_object::Flattener;
use serde::{Deserialize, Serialize};
use serde_bytes::ByteBuf;
use serde_json::{json, Value};

use crate::cert::dice::cbor::{
    HashAlgoIanaId, OCBR_TAG_EVIDENCE_COCO_EVIDENCE, OCBR_TAG_EVIDENCE_COCO_TOKEN,
};
use crate::crypto::HashAlgo;
use crate::errors::*;
use crate::tee::claims::Claims;
use crate::tee::{DiceParseEvidenceOutput, GenericEvidence};

pub(crate) struct AaTeeType(String);

impl AaTeeType {
    pub fn as_attestation_service_str_id(&self) -> &str {
        match self.0.as_str() {
            // AA is using "sgx-dcap" while "sgx" is required in AS. See https://github.com/confidential-containers/trustee/blob/09bef2e2a53d54c2d3107635a65337f409eeaebe/attestation-service/attestation-service/src/bin/grpc/mod.rs#L29-L41
            "sgx-dcap" => "sgx",
            v => v,
        }
    }

    pub fn as_attestation_agent_str_id(&self) -> &str {
        &self.0
    }

    pub fn from_attestation_agent_str_id(tee_type_str: impl Into<String>) -> Self {
        Self(tee_type_str.into())
    }
}

pub struct CocoEvidence {
    pub(self) aa_tee_type: AaTeeType,
    /// Raw evidence generated by attestation-agent
    pub(self) aa_evidence: Vec<u8>,
    /// The structed runtime data json which is passed to attestation-agent for generating the evidence, should be hashed with aa_runtime_data_algo.
    pub(self) aa_runtime_data: String,
    /// The hash algorithm used for calculating hash of aa_runtime_data.
    pub(self) aa_runtime_data_hash_algo: HashAlgo,
}

impl CocoEvidence {
    pub(crate) fn new(
        aa_tee_type: AaTeeType,
        aa_evidence: Vec<u8>,
        aa_runtime_data: String,
        aa_runtime_data_hash_algo: HashAlgo,
    ) -> Result<Self> {
        Ok(Self {
            aa_tee_type,
            aa_evidence,
            aa_runtime_data,
            aa_runtime_data_hash_algo,
        })
    }

    pub(crate) fn get_tee_type(&self) -> &AaTeeType {
        &self.aa_tee_type
    }

    pub(crate) fn aa_evidence_ref(&self) -> &[u8] {
        &self.aa_evidence
    }

    pub(crate) fn aa_runtime_data_ref(&self) -> &str {
        &self.aa_runtime_data
    }

    pub(crate) fn get_aa_runtime_data_hash_algo(&self) -> HashAlgo {
        self.aa_runtime_data_hash_algo
    }

    pub(crate) fn wrap_runtime_data_as_structed(report_data: &[u8]) -> Result<String> {
        serde_json::to_string(
            &json!({"rats-rs.raw_runtime_data": URL_SAFE_NO_PAD.encode(report_data)}),
        )
        .context("Failed to serialize structed runtime data")
    }
}

impl GenericEvidence for CocoEvidence {
    fn get_dice_cbor_tag(&self) -> u64 {
        return OCBR_TAG_EVIDENCE_COCO_EVIDENCE;
    }

    fn get_dice_raw_evidence(&self) -> Result<Vec<u8>> {
        let mut res = vec![];
        ciborium::into_writer(&self, &mut res)?;
        Ok(res)
    }

    fn get_claims(&self) -> Result<Claims> {
        // Return empty claims here since the structure of AA evidence is opaque.
        Ok(Claims::default())
    }

    fn create_evidence_from_dice(
        cbor_tag: u64,
        raw_evidence: &[u8],
    ) -> DiceParseEvidenceOutput<Self> {
        if cbor_tag == OCBR_TAG_EVIDENCE_COCO_EVIDENCE {
            return match ciborium::from_reader(raw_evidence)
                .context("Failed to deserialize coco evidence")
            {
                Ok(v) => DiceParseEvidenceOutput::Ok(v),
                Err(e) => DiceParseEvidenceOutput::MatchButInvalid(e.into()),
            };
        }
        return DiceParseEvidenceOutput::NotMatch;
    }
}

pub struct CocoAsToken {
    data: String,
}

impl CocoAsToken {
    pub(crate) fn new(token: String) -> Result<Self> {
        Ok(Self { data: token })
    }

    pub(crate) fn as_str(&self) -> &str {
        &self.data
    }
}

impl GenericEvidence for CocoAsToken {
    fn get_dice_cbor_tag(&self) -> u64 {
        return OCBR_TAG_EVIDENCE_COCO_TOKEN;
    }

    fn get_dice_raw_evidence(&self) -> Result<Vec<u8>> {
        Ok(self.data.as_bytes().to_owned())
    }

    fn get_claims(&self) -> Result<Claims> {
        let split_token: Vec<&str> = self.data.split('.').collect();
        if !split_token.len() == 3 {
            return Err(Error::kind_with_msg(
                ErrorKind::CocoVerifyTokenFailed,
                "Illegal JWT format",
            ));
        }
        let claims = URL_SAFE_NO_PAD.decode(split_token[1])?;
        let claims_value = serde_json::from_slice::<Value>(&claims)?;

        let flattened_claims_value = Flattener::new()
            .flatten(&claims_value)
            .context("Failed to flatten JWT claims JSON object")?;

        Ok(match flattened_claims_value {
            Value::Object(m) => m
                .into_iter()
                .filter_map(|(k, v)| -> Option<_> {
                    match v {
                        Value::String(s) => Some((k, s.as_bytes().into())),
                        Value::Null => None,
                        v => Some((k, v.to_string().into())),
                    }
                })
                .collect::<Claims>(),
            _ => {
                return Err(Error::kind_with_msg(
                    ErrorKind::CocoParseTokenFailed,
                    format!("Invalid claims value: {}", claims_value),
                ))
            }
        })
    }

    fn create_evidence_from_dice(
        cbor_tag: u64,
        raw_evidence: &[u8],
    ) -> DiceParseEvidenceOutput<Self> {
        if cbor_tag == OCBR_TAG_EVIDENCE_COCO_TOKEN {
            return match std::str::from_utf8(raw_evidence)
                .context("Failed to parse evidence as utf-8 string")
                .map(|token| Self::new(token.to_owned()))
            {
                Ok(Ok(v)) => DiceParseEvidenceOutput::Ok(v),
                Ok(Err(e)) => DiceParseEvidenceOutput::MatchButInvalid(e),
                Err(e) => DiceParseEvidenceOutput::MatchButInvalid(e.into()),
            };
        }
        return DiceParseEvidenceOutput::NotMatch;
    }
}

#[derive(Serialize, Deserialize)]
struct CocoEvidenceCborHelper {
    pub(self) tee_type: String,
    pub(self) aa_evidence: ByteBuf,
    pub(self) aa_runtime_data: String,
    pub(self) aa_runtime_data_hash_algo: HashAlgoIanaId,
}

impl Serialize for CocoEvidence {
    fn serialize<S>(&self, serializer: S) -> std::result::Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let helper = CocoEvidenceCborHelper {
            tee_type: self.aa_tee_type.as_attestation_agent_str_id().to_owned(),
            aa_evidence: ByteBuf::from(self.aa_evidence.to_owned()),
            aa_runtime_data: self.aa_runtime_data.to_owned(),
            aa_runtime_data_hash_algo: self.aa_runtime_data_hash_algo.into(),
        };

        helper.serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for CocoEvidence {
    fn deserialize<D>(deserializer: D) -> std::prelude::v1::Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let helper = CocoEvidenceCborHelper::deserialize(deserializer)?;

        Ok(Self {
            aa_tee_type: AaTeeType::from_attestation_agent_str_id(&helper.tee_type),
            aa_evidence: helper.aa_evidence.into_vec(),
            aa_runtime_data: helper.aa_runtime_data,
            aa_runtime_data_hash_algo: helper
                .aa_runtime_data_hash_algo
                .try_into()
                .map_err(|e| serde::de::Error::custom(format!("{e:?}")))?,
        })
    }
}
